﻿unit Artifacts_Mobile;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  ByteStream, Classes, Clipbrd, Common, Contnrs, DataEntry, DataStorage, DateUtils, DIRegex,
  Graphics, JSON, PropertyList, Math, ProtoBuf, Regex, SQLite3, SysUtils, Variants, XML,
  Artifact_Utils      in 'Common\Artifact_Utils.pas',
  Columns             in 'Common\Column_Arrays\Columns.pas',
  Icon_List           in 'Common\Icon_List.pas',
  Itunes_Backup_Sig   in 'Common\Itunes_Backup_Signature_Analysis.pas',
  Phone_MMC_DB        in 'Common\Phone_MMC_DB.pas',
  RS_DoNotTranslate   in 'Common\RS_DoNotTranslate.pas',
  Mobile_Columns      in 'Common\Column_Arrays\Arrays_Mobile.pas';

type
  TDataStoreFieldArray = array of TDataStoreField;

  TSQL_FileSearch = record
    fi_Carve_Adjustment           : integer;
    fi_Carve_Footer               : string;
    fi_Carve_Header               : string;
    fi_Glob1_Search               : string;
    fi_Glob2_Search               : string;
    fi_Glob3_Search               : string;
    fi_Icon_Category              : integer;
    fi_Icon_OS                    : integer;
    fi_Icon_Program               : integer;
    fi_Name_OS                    : string;
    fi_Name_Program               : string;
    fi_Name_Program_Type          : string;
    fi_NodeByName                 : string;
    fi_Process_As                 : string;
    fi_Process_ID                 : string;
    fi_Reference_Info             : string;
    fi_Regex_Search               : string;
    fi_Rgx_Itun_Bkup_Dmn          : string;
    fi_Rgx_Itun_Bkup_Nme          : string;
    fi_RootNodeName               : string;
    fi_Signature_Parent           : string;
    fi_Signature_Sub              : string;
    fi_SQLPrimary_Tablestr        : string;
    fi_SQLStatement               : string;
    fi_SQLTables_Required         : string;
    fi_Test_Data                  : string;
  end;

  PSQL_FileSearch = ^TSQL_FileSearch;
  TgArr = array of TSQL_FileSearch;

const
  ARTIFACTS_DB              = 'Artifacts_Mobile.db';
  CATEGORY_NAME             = 'Mobile';
  PROGRAM_NAME              = 'Mobile';
  SCRIPT_DESCRIPTION        = 'Extract Mobile Artifacts';
  SCRIPT_NAME               = 'Mobile.pas';
  // ~~~
  ATRY_EXCEPT_STR           = 'TryExcept: '; // noslz
  ANDROID                   = 'Android'; // noslz
  BL_BOOKMARK_SOURCE        = False;
  BL_PROCEED_LOGGING        = False;
  BL_USE_FLAGS              = False;
  BS                        = '\';
  CANCELED_BY_USER          = 'Canceled by user.';
  CHAR_LENGTH               = 80;
  COLON                     = ':';
  COMMA                     = ',';
  CR                        = #13#10;
  DCR                       = #13#10 + #13#10;
  DUP_FILE                  = '( \d*)?$';
  FBN_ITUNES_BACKUP_DOMAIN  = 'iTunes Backup Domain'; // noslz
  FBN_ITUNES_BACKUP_NAME    = 'iTunes Backup Name'; // noslz
  GFORMAT_STR               = '%-1s %-12s %-8s %-15s %-25s %-30s %-20s';
  GROUP_NAME                = 'Artifact Analysis';
  HYPHEN                    = ' - ';
  HYPHEN_NS                 = '-';
  ICON_ANDROID              = 1017;
  ICON_IOS                  = 1062;
  IOS                       = 'iOS'; // noslz
  MAX_CARVE_SIZE            = 1024 * 10;
  MIN_CARVE_SIZE            = 0;
  PHOTOS_SQLITE             = 'Photos.Sqlite'; // noslz
  PIPE                      = '|'; // noslz
  PROCESS_AS_BYTES_STRLST   = 'PROCESS_AS_BYTES_STRLST'; // noslz
  PROCESS_AS_CARVE          = 'PROCESSASCARVE'; // noslz
  PROCESS_AS_PLIST          = 'PROCESSASPLIST'; // noslz
  PROCESS_AS_TEXT           = 'PROCESSASTEXT'; // noslz
  PROCESS_AS_XML            = 'PROCESSASXML'; // noslz
  PROCESSALL                = 'PROCESSALL'; // noslz
  RPAD_VALUE                = 55;
  RUNNING                   = '...';
  SPACE                     = ' ';
  STR_FILES_BY_PATH         = 'Files by path';
  TRIAGE                    = 'TRIAGE'; // noslz
  TSWT                      = 'The script will terminate.';
  VERBOSE                   = False;
  WINDOWS                   = 'Windows'; // noslz

var
  gArr:                           TgArr;
  gArr_ValidatedFiles_TList:      array of TList;
  gArtConnect_CatFldr:            TArtifactEntry1;
  gArtConnect_ProgFldr:           array of TArtifactConnectEntry;
  gArtifacts_SQLite:              TSQLite3Database;
  gArtifactsDataStore:            TDataStore;
  gdb_read_bl:                    boolean;
  gFileSystemDataStore:           TDataStore;
  gNumberOfSearchItems:           integer;
  gParameter_Num_StringList:      TStringList;
  gtick_doprocess_i64:            uint64;
  gtick_doprocess_str:            string;
  gtick_foundlist_i64:            uint64;
  gtick_foundlist_str:            string;

function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
function FileSubSignatureMatch(Entry: TEntry): boolean;
function GetFullName(Item: PSQL_FileSearch): string;
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
function RPad(const AString: string; AChars: integer): string;
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
function TestForDoProcess(ARefNum: integer): boolean;
function TotalValidatedFileCountInTLists: integer;
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);

{$IF DEFINED (ISFEXGUI)}
type
  TScriptForm = class(TObject)
  private
    frmMain: TGUIForm;
    FMemo: TGUIMemoBox;
    pnlBottom: TGUIPanel;
    btnOK: TGUIButton;
    btnCancel: TGUIButton;
  public
    ModalResult: boolean;
    constructor Create;
    function ShowModal: boolean;
    procedure OKClick(Sender: TGUIControl);
    procedure CancelClick(Sender: TGUIControl);
    procedure SetText(const Value: string);
    procedure SetCaption(const Value: string);
  end;
{$IFEND}

implementation

{$IF DEFINED (ISFEXGUI)}

constructor TScriptForm.Create;
begin
  inherited Create;
  frmMain := NewForm(nil, SCRIPT_NAME);
  frmMain.Size(500, 480);
  pnlBottom := NewPanel(frmMain, esNone);
  pnlBottom.Size(frmMain.Width, 40);
  pnlBottom.Align(caBottom);
  btnOK := NewButton(pnlBottom, 'OK');
  btnOK.Position(pnlBottom.Width - 170, 4);
  btnOK.Size(75, 25);
  btnOK.Anchor(False, True, True, True);
  btnOK.DefaultBtn := True;
  btnOK.OnClick := OKClick;
  btnCancel := NewButton(pnlBottom, 'Cancel');
  btnCancel.Position(pnlBottom.Width - 88, 4);
  btnCancel.Size(75, 25);
  btnCancel.Anchor(False, True, True, True);
  btnCancel.CancelBtn := True;
  btnCancel.OnClick := CancelClick;
  FMemo := NewMemoBox(frmMain, [eoReadOnly]); // eoNoHScroll, eoNoVScroll
  FMemo.Color := clWhite;
  FMemo.Align(caClient);
  FMemo.FontName('Courier New');
  frmMain.CenterOnParent;
  frmMain.Invalidate;
end;

procedure TScriptForm.OKClick(Sender: TGUIControl);
begin
  // Set variables for use in the main proc. Must do this before closing the main form.
  ModalResult := False;
  ModalResult := True;
  frmMain.Close;
end;

procedure TScriptForm.CancelClick(Sender: TGUIControl);
begin
  ModalResult := False;
  Progress.DisplayMessageNow := CANCELED_BY_USER;
  frmMain.Close;
end;

function TScriptForm.ShowModal: boolean;
begin
  Execute(frmMain);
  Result := ModalResult;
end;

procedure TScriptForm.SetText(const Value: string);
begin
  FMemo.Text := Value;
end;

procedure TScriptForm.SetCaption(const Value: string);
begin
  frmMain.Text := Value;
end;
{$IFEND}

// -----------------------------------------------------------------------------
// Add 40 Character Files
// -----------------------------------------------------------------------------
procedure Add40CharFiles(UniqueList: TUniqueListOfEntries);
var
  Entry: TEntry;
  FileSystemDS: TDataStore;
begin
  FileSystemDS := GetDataStore(DATASTORE_FILESYSTEM);
  if assigned(UniqueList) and assigned(FileSystemDS) then
  begin
    Entry := FileSystemDS.First;
    while assigned(Entry) and Progress.isRunning do
    begin
      begin
        if RegexMatch(Entry.EntryName, '^[0-9a-fA-F]{40}', False) and (Entry.EntryNameExt = '') then // noslz iTunes Backup files
        begin
          UniqueList.Add(Entry);
        end;
      end;
      Entry := FileSystemDS.Next;
    end;
    FreeAndNil(FileSystemDS);
  end;
end;

// -----------------------------------------------------------------------------
// Apple Maps Protobuf
// -----------------------------------------------------------------------------
function AppleMaps_ProtoBuf(bytes: TBytes; type_str: string) : string;
const
  {ZHISTORYITEM.ZROUTEREQUESTSTORAGE AS 'Journey BLOB'} // noslz
  {ZMIXINMAPITEM.ZMAPITEMSTORAGE as 'Map Item Storage BLOB'} // noslz
  CL = 5;
  VERBOSE = False;
var
  array_of_cells: TArrayCells;
  array_of_cells_8: TArrayCells;
  cp: Cellptr;
  cp2: Cellptr;
  cp8: Cellptr;
  i: integer;
  k: integer;
  pbr: TGDProtoBufReader;
  row8_str: string;
  row9_str: string;
  str: string;

  function clean_str(astr: string) : string;
  begin
    Result := aStr;
    Result := StringReplace(Result, '","', ', ', [rfReplaceAll]);
    Result := StringReplace(Result, '"', '', [rfReplaceAll]);
  end;

begin
  Result := '';
  row8_str := '';
  row9_str := '';
  pbr := TGDProtoBufReader.Create;
  try
    pbr.Loadfrombytes(bytes);

    if type_str = 'ZROUTEREQUESTSTORAGE' then // noslz
    begin
      str := pbr.ProBufLookupValue(pbr.rootcell, [1.2, 1, 2], [6, 6.2, 6.3]);
      Result := clean_str(str);
      // Progress.Log(RPad(HYPHEN + type_str, RPAD_VALUE) + Result);
    end;

    if type_str = 'ZMAPITEMSTORAGE' then // noslz
    begin
      array_of_cells := pbr.ProBufLookupList(pbr.rootcell, [1], 4); // Comes back with a list of all the field 4'a
      try
        // Outer Loop
        i := 0;
        while i < length(array_of_cells) do // Loops over the array_of_cells
        begin
          cp := array_of_cells[i];
          array_of_cells_8 := pbr.ProBufLookupList(cp, [], 8);
          try
            // Inner Loop
            for k := 0 to length(array_of_cells_8) - 1 do // Loop over the second array_of_cells
            begin
              cp8 := array_of_cells_8[k];
              cp2 := pbr.ProBufLookupfield(cp8, [31, 1, 101, 2]);
              if cp2 <> nil then
              begin
                str := pbr.ProBufLookupValue(cp2, [], [11, 11.2, 11.3]);
                if str <> '' then
                begin
                  Result := clean_str(str);
                  // Progress.Log(RPad(HYPHEN + type_str, RPAD_VALUE) + Result);
                  i := length(array_of_cells); // Break the outer loop
                  break; // Break the inner loop
                end;
              end;
            end;
            inc(i);
          finally
            array_of_cells_8 := nil; // Free the array
          end;
        end;
      finally
        array_of_cells := nil; // Free the array
      end;
    end;

  finally
    pbr.free;
  end;
end;

// -----------------------------------------------------------------------------
// Bookmark Artifact
// -----------------------------------------------------------------------------
procedure Bookmark_Artifact_Source(Entry: TEntry; name_str: string; category_str: string; bm_comment_str: string);
var
  bmDeviceEntry: TEntry;
  bmCreate: boolean;
  bmFolder: TEntry;
  device_name_str: string;
begin
  bmCreate := True;
  if assigned(Entry) then
  begin
    bmDeviceEntry := GetDeviceEntry(Entry);
    device_name_str := '';
    if assigned(bmDeviceEntry) then device_name_str := bmDeviceEntry.SaveName + BS;
    bmFolder := FindBookmarkByName('Artifacts' + BS + 'Source Files' + BS {+ device_name_str + BS} + category_str + BS + name_str, bmCreate);
    if assigned(bmFolder) then
    begin
      if not IsItemInBookmark(bmFolder, Entry) then
      begin
        AddItemsToBookmark(bmFolder, DATASTORE_FILESYSTEM, Entry, bm_comment_str);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Column Value By Name As Date Time
// -----------------------------------------------------------------------------
function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
var
  iCol: integer;
begin
  Result := 0;
  iCol := ColumnByName(Statement, copy(colConf.sql_col, 5, Length(colConf.sql_col)));
  if (iCol > -1) then
  begin
    if colConf.read_as = ftLargeInt then
    try
      Result := Int64ToDateTime_ConvertAs(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end
    else if colConf.read_as = ftFloat then
    try
      Result := GHFloatToDateTime(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Create Artifact Description Text
// -----------------------------------------------------------------------------
function Create_Artifact_Description_Text(Item: TSQL_FileSearch): string;
var
  aStringList: TStringList;

  procedure AddIntToSL(const name_str: string; aint: integer);
  begin
    if aint > 0 then
      aStringList.Add(RPad(name_str + ':', 25) + IntToStr(aint));
  end;

  procedure AddStrToSL(const name_str: string;const astr: string);
  begin
    if Trim(astr) <> '' then
      aStringList.Add(RPad(name_str + ':', 25) + astr);
  end;

begin
  Result := '';
  aStringList := TStringList.Create;
  try
    AddStrToSL('Process_ID', Item.fi_Process_ID);
    AddStrToSL('Name_Program', Item.fi_Name_Program);
    AddStrToSL('Name_Program_Type', Item.fi_Name_Program_Type);
    AddStrToSL('Name_OS', Item.fi_Name_OS);
    AddStrToSL('Process_As', Item.fi_Process_As);
    AddIntToSL('Carve_Adjustment', Item.fi_Carve_Adjustment);
    AddStrToSL('Carve_Footer', Item.fi_Carve_Footer);
    AddStrToSL('Carve_Header', Item.fi_Carve_Header);
    AddIntToSL('Icon_Category', Item.fi_Icon_Category);
    AddIntToSL('Icon_OS', Item.fi_Icon_OS);
    AddIntToSL('Icon_Program', Item.fi_Icon_Program);
    AddStrToSL('NodeByName', Item.fi_NodeByName);
    AddStrToSL('Reference_Info', Item.fi_Reference_Info);
    AddStrToSL('Rgx_Itun_Bkup_Dmn', Item.fi_Rgx_Itun_Bkup_Dmn);
    AddStrToSL('Rgx_Itun_Bkup_Nme', Item.fi_Rgx_Itun_Bkup_Nme);
    AddStrToSL('Glob1_Search', Item.fi_Glob1_Search);
    AddStrToSL('Glob2_Search', Item.fi_Glob2_Search);
    AddStrToSL('Glob3_Search', Item.fi_Glob3_Search);
    AddStrToSL('Regex_Search', Item.fi_Regex_Search);
    AddStrToSL('RootNodeName', Item.fi_RootNodeName);
    AddStrToSL('Signature_Parent', Item.fi_Signature_Parent);
    AddStrToSL('Signature_Sub', Item.fi_Signature_Sub);
    AddStrToSL('SQLStatement', Item.fi_SQLStatement);
    AddStrToSL('fi_SQLTables_Required', Item.fi_SQLTables_Required);
    AddStrToSL('SQLPrimary_Tablestr', Item.fi_SQLPrimary_Tablestr);
    Result := aStringList.Text;
  finally
    aStringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// Create Global Search
// -----------------------------------------------------------------------------
procedure Create_Global_Search(anint: integer; Item: TSQL_FileSearch; aStringList: TStringList);
var
  Glob1_Search: string;
  Glob2_Search: string;
  Glob3_Search: string;
begin
  Glob1_Search := Item.fi_Glob1_Search;
  Glob2_Search := Item.fi_Glob2_Search;
  Glob3_Search := Item.fi_Glob3_Search;
  if Trim(Glob1_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob1_Search + ''');'); // noslz
  if Trim(Glob2_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob2_Search + ''');'); // noslz
  if Trim(Glob3_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob3_Search + ''');'); // noslz
  if Trim(Glob1_Search) =  '' then Progress.Log(IntTostr(anint) + ': No Glob Search'); // noslz
end;

// -----------------------------------------------------------------------------
// Determine Then Skip Or Add
// -----------------------------------------------------------------------------
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
var
  bm_comment_str: string;
  bmwal_Entry: TEntry;
  DeterminedFileDriverInfo: TFileTypeInformation;
  File_Added_bl: boolean;
  first_Entry: TEntry;
  i: integer;
  Item: TSQL_FileSearch;
  MatchSignature_str: string;
  NowProceed_bl: boolean;
  reason_str: string;
  Reason_StringList: TStringList;
  trunc_EntryName_str: string;

begin
  File_Added_bl := False;
  if Entry.isSystem and ((POS('$I30', Entry.EntryName) > 0) or (POS('$90', Entry.EntryName) > 0)) then
  begin
    NowProceed_bl := False;
    Exit;
  end;

  DeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
  reason_str := '';
  trunc_EntryName_str := copy(Entry.EntryName, 1, 25);
  Reason_StringList := TStringList.Create;
  try
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      NowProceed_bl := False;
      reason_str := '';
      Item := gArr[i];

      // -----------------------------------------------------------------------
      // Special validation (these files are still sent here via the Regex match)
      // -----------------------------------------------------------------------
      if (not NowProceed_bl) then
      begin
        if RegexMatch(Entry.EntryName, gArr[i].fi_Regex_Search, False) and
          (RegexMatch(Entry.EntryName, 'History.mapsdata', False) or
          RegexMatch(Entry.EntryName, '1321e6b74c9dfe411e7e129d6a8ae7cc645af9d0', False) or
          RegexMatch(Entry.EntryName, '^pass\.json', False) or
          RegexMatch(Entry.EntryName, 'flattened-data', False)) then
          NowProceed_bl := True;
      end;

      // Proceed if SubDriver has been identified
      if (not NowProceed_bl) then
      begin
        if Item.fi_Signature_Sub <> '' then
        begin
          if RegexMatch(RemoveSpecialChars(DeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then
          begin
            NowProceed_bl := True;
            reason_str := 'ShortDisplayName = Required SubSig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + Item.fi_Signature_Sub + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(A)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;
      end;

      // Set the MatchSignature to the parent
      if (not NowProceed_bl) and not(UpperCase(Entry.Extension) = '.JSON') then
      begin
        MatchSignature_str := '';
        if Item.fi_Signature_Sub = '' then
          MatchSignature_str := UpperCase(Item.fi_Signature_Parent);
        // Proceed if SubDriver is blank, but File/Path Name and Parent Signature match
        if ((RegexMatch(Entry.EntryName, Item.fi_Regex_Search, False)) or (RegexMatch(Entry.FullPathName, Item.fi_Regex_Search, False))) and
          ((UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) or (RegexMatch(DeterminedFileDriverInfo.ShortDisplayName, MatchSignature_str, False))) then
        begin
          NowProceed_bl := True;
          reason_str := 'ShortDisplay matches Parent sig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(B)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      // Proceed if EntryName is unknown, but iTunes Domain, Name and Sig match
      if (not NowProceed_bl) then
      begin
        if RegexMatch(biTunes_Domain_str, Item.fi_Rgx_Itun_Bkup_Dmn, False) and RegexMatch(biTunes_Name_str, Item.fi_Rgx_Itun_Bkup_Nme, False) and (UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) then
        begin
          NowProceed_bl := True;
          reason_str := 'Proceed on Sig and iTunes Domain\Name:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(C)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      if NowProceed_bl then
      begin
        gArr_ValidatedFiles_TList[i].Add(Entry);
        File_Added_bl := True;

        // Bookmark Source Files
        if BL_BOOKMARK_SOURCE then
        begin
          bm_comment_str := Create_Artifact_Description_Text(gArr[i]);
          Bookmark_Artifact_Source(Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
          if assigned(gFileSystemDataStore) then
          begin
            first_Entry := gFileSystemDataStore.First;
            if assigned(first_Entry) then
            begin
              bmwal_Entry := gFileSystemDataStore.FindByPath(first_Entry, Entry.FullPathName + '-wal');
              if bmwal_Entry <> nil then
              begin
                Bookmark_Artifact_Source(bmwal_Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
              end;
            end;
          end;
        end;

        // Add Flags
        if BL_USE_FLAGS then Entry.Flags := Entry.Flags + [Flag5]; // Green Flag
      end;
    end;

    if NOT(File_Added_bl) then
      Reason_StringList.Add(format(GFORMAT_STR, ['', 'Ignored', '', IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]));

    for i := 0 to Reason_StringList.Count - 1 do
      Progress.Log(Reason_StringList[i]);

  finally
    Reason_StringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// File Sub Signature Match
// -----------------------------------------------------------------------------
function FileSubSignatureMatch(Entry: TEntry): boolean;
var
  i: integer;
  param_num_int: integer;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  Item: TSQL_FileSearch;
begin
  Result := False;
  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
  begin
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      Item := gArr[i];
      if Item.fi_Signature_Sub <> '' then
      begin
        aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
        if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
        begin
          if BL_PROCEED_LOGGING then
            Progress.Log(RPad('Proceed' + HYPHEN + 'Identified by SubSig:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
          Result := True;
          break;
        end;
      end;
    end;
  end
  else
  begin
    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
    begin
      for i := 0 to gParameter_Num_StringList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        param_num_int := StrToInt(gParameter_Num_StringList[i]);
        Item := gArr[param_num_int];
        if Item.fi_Signature_Sub <> '' then
        begin
          aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
          if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
          begin
            if BL_PROCEED_LOGGING then
              Progress.Log(RPad('Proceed' + HYPHEN + 'File Sub-Signature Match:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
            Result := True;
            break;
          end;
        end;
      end
    end
  end;
end;

// -----------------------------------------------------------------------------
// Get Full Name
// -----------------------------------------------------------------------------
function GetFullName(Item: PSQL_FileSearch): string;
var
  ApplicationName: string;
  TypeName: string;
  OSName: string;
begin
  Result := '';
  ApplicationName := Item.fi_Name_Program;
  TypeName := Item.fi_Name_Program_Type;
  OSName := Item.fi_Name_OS;
  if (ApplicationName <> '') then
  begin
    if (TypeName <> '') then
      Result := format('%0:s %1:s', [ApplicationName, TypeName])
    else
      Result := ApplicationName;
  end
  else
    Result := TypeName;
  if OSName <> '' then
    Result := Result + ' ' + OSName;
end;

// -----------------------------------------------------------------------------
// Length of Array Table
// -----------------------------------------------------------------------------
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to 100 do
  begin
    if anArray[i].sql_col = '' then
      break;
    Result := i;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// -----------------------------------------------------------------------------
// Setup Column For Folder
// -----------------------------------------------------------------------------
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
var
  col_label: string;
  col_source_created: TDataStoreField;
  col_source_file: TDataStoreField;
  col_source_modified: TDataStoreField;
  col_source_path: TDataStoreField;
  Field: TDataStoreField;
  i: integer;
  Item: TSQL_FileSearch;
  NumberOfColumns: integer;

begin
  Result := True;
  Item := gArr[aReferenceNumber];
  NumberOfColumns := ColCount;
  SetLength(col_DF, ColCount + 1);

  if assigned(anArtifactFolder) then
  begin
    for i := 1 to NumberOfColumns do
    begin
      try
        if not Progress.isRunning then
          Exit;
        Field := gArtifactsDataStore.DataFields.FieldByName(aItems[i].fex_col);
        if assigned(Field) and (Field.FieldType <> aItems[i].col_type) then
        begin
          MessageUser(SCRIPT_NAME + DCR + 'WARNING: New column: ' + DCR + aItems[i].fex_col + DCR + 'already exists as a different type. Creation skipped.');
          Result := False;
        end
        else
        begin
          col_label := '';
          col_DF[i] := gArtifactsDataStore.DataFields.Add(aItems[i].fex_col + col_label, aItems[i].col_type);
          if col_DF[i] = nil then
          begin
            MessageUser(SCRIPT_NAME + DCR + 'Cannot use a fixed field. Please contact support@getdata.com quoting the following error: ' + DCR + SCRIPT_NAME + SPACE + IntToStr(aReferenceNumber) + SPACE + aItems[i].fex_col);
            Result := False;
          end;
        end;
      except
        MessageUser(ATRY_EXCEPT_STR + 'Failed to create column');
      end;
    end;

    // Set the Source Columns --------------------------------------------------
    col_source_file := gArtifactsDataStore.DataFields.GetFieldByName('Source_Name');
    col_source_path := gArtifactsDataStore.DataFields.GetFieldByName('Source_Path');
    col_source_created := gArtifactsDataStore.DataFields.GetFieldByName('Source_Created');
    col_source_modified := gArtifactsDataStore.DataFields.GetFieldByName('Source_Modified');

    // Columns -----------------------------------------------------------------
    if Result then
    begin
      // Enables the change of column headers when switching folders - This is the order of displayed columns
      for i := 1 to NumberOfColumns do
      begin
        if not Progress.isRunning then
          break;
        if aItems[i].Show then
        begin
          // Progress.Log('Add Field Name: ' + col_DF[i].FieldName);
          anArtifactFolder.AddField(col_DF[i]);
        end;
      end;

      if (Item.fi_Process_As = 'POSTPROCESS') then
        anArtifactFolder.AddField(col_source_path)
      else
      begin
        anArtifactFolder.AddField(col_source_file);
        anArtifactFolder.AddField(col_source_path);
        anArtifactFolder.AddField(col_source_created);
        anArtifactFolder.AddField(col_source_modified);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Name
// -----------------------------------------------------------------------------
function SQLColumnByName(Statement: TSQLite3Statement; const Name: string): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Statement.ColumnCount do
  begin
    if not Progress.isRunning then
      break;
    if SameText(Statement.ColumnName(i), Name) then
    begin
      Result := i;
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Integer
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsInt(Statement: TSQLite3Statement; const Name: string): integer;
var
  iCol: integer;
begin
  Result := -1;
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnInt(iCol);
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Text
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsText(Statement: TSQLite3Statement; const Name: string): string;
var
  iCol: integer;
begin
  Result := '';
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnText(iCol);
end;

// -----------------------------------------------------------------------------
// SQL - Read Artifacts DB
// -----------------------------------------------------------------------------
procedure Read_SQLite_DB();
var
  i: integer;
  sql_db_path_str: string;
  sqlselect: TSQLite3Statement;

begin
  // Locate the SQLite DB
  sql_db_path_str := GetDatabasesDir + 'Artifacts' + BS + ARTIFACTS_DB; // noslz
  if not FileExists(sql_db_path_str) then
  begin
    MessageUser('Mobile' + ':' + SPACE + 'Did not locate Artifacts SQLite Database:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
    Exit;
  end;
  Progress.Log(RPad('Found Database:', RPAD_VALUE) + sql_db_path_str);

  // Open the database
  gArtifacts_SQLite := TSQLite3Database.Create;
  try
    gdb_read_bl := False;
    if FileExists(sql_db_path_str) then
    try
      gArtifacts_SQLite.Open(sql_db_path_str);
      gdb_read_bl := True;
      Progress.Log(RPad('Database Read:', RPAD_VALUE) + BoolToStr(gdb_read_bl, True));
    except
      on e: exception do
      begin
        Progress.Log(e.message);
        Exit;
      end;
    end;

    // Get the number of rows in the database as gNumberOfSearchItems
    if gdb_read_bl then
    begin
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT COUNT(*) FROM Artifact_Values');
      try
        gNumberOfSearchItems := 0;
        if sqlselect.Step = SQLITE_ROW then
          gNumberOfSearchItems := sqlselect.ColumnInt(0);
      finally
        sqlselect.free;
      end;
      Progress.Log(RPad('Database Rows:', RPAD_VALUE) + IntToStr(gNumberOfSearchItems));

      if gNumberOfSearchItems = 0 then
      begin
        MessageUser('No records were read from:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
        Exit;
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      SetLength(gArr, gNumberOfSearchItems);

      // Populate the Array with values from the database
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT * FROM Artifact_Values ORDER BY fi_Name_Program');
      try
        i := 0;
        while (sqlselect.Step = SQLITE_ROW) and (Progress.isRunning) do
        begin
          gArr[i].fi_Carve_Adjustment    := SQLColumnValueByNameAsInt(sqlselect,  'fi_Carve_Adjustment');
          gArr[i].fi_Carve_Footer        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Footer');
          gArr[i].fi_Carve_Header        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Header');
          gArr[i].fi_Glob1_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob1_Search');
          gArr[i].fi_Glob2_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob2_Search');
          gArr[i].fi_Glob3_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob3_Search');
          gArr[i].fi_Icon_Category       := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Category');
          gArr[i].fi_Icon_OS             := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_OS');
          gArr[i].fi_Icon_Program        := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Program');
          gArr[i].fi_Name_Program        := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program');
          gArr[i].fi_Name_OS             := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_OS');
          gArr[i].fi_Name_Program_Type   := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program_Type');
          gArr[i].fi_NodeByName          := SQLColumnValueByNameAsText(sqlselect, 'fi_NodeByName');
          gArr[i].fi_Process_As          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_As');
          gArr[i].fi_Process_ID          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_ID');
          gArr[i].fi_Reference_Info      := SQLColumnValueByNameAsText(sqlselect, 'fi_Reference_Info');
          gArr[i].fi_Regex_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Regex_Search');
          gArr[i].fi_Rgx_Itun_Bkup_Dmn   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Dmn');
          gArr[i].fi_Rgx_Itun_Bkup_Nme   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Nme');
          gArr[i].fi_RootNodeName        := SQLColumnValueByNameAsText(sqlselect, 'fi_RootNodeName');
          gArr[i].fi_Signature_Parent    := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Parent');
          gArr[i].fi_Signature_Sub       := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Sub');
          gArr[i].fi_SQLPrimary_Tablestr := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLPrimary_Tablestr');
          gArr[i].fi_SQLStatement        := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLStatement');
          gArr[i].fi_SQLTables_Required  := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLTables_Required');
          gArr[i].fi_Test_Data           := SQLColumnValueByNameAsText(sqlselect, 'fi_Test_Data');
          inc(i);
        end;

      finally
        sqlselect.free;
      end;

    end;

  finally
    gArtifacts_SQLite.Close;
    gArtifacts_SQLite.free;
  end;
end;

// -----------------------------------------------------------------------------
// StrippedOfNonAscii
// -----------------------------------------------------------------------------
function StrippedOfNonAscii(const str: string): string;
var
  idx, Count: integer;
begin
  SetLength(Result, Length(str));
  Count := 0;
  for idx := 1 to Length(str) do
  begin
    if ((str[idx] >= #32) and (str[idx] <= #127)) or (str[idx] in [#10, #13]) then
    begin
      Inc(Count);
      Result[Count] := str[idx];
    end;
  end;
  SetLength(Result, Count);
end;

// -----------------------------------------------------------------------------
// Test for Do Process
// -----------------------------------------------------------------------------
function TestForDoProcess(ARefNum: integer): boolean;
begin
  Result := False;

  if gArr[ARefNum].fi_Process_As = 'POSTPROCESS' then
  begin
    Result := True;
    Exit;
  end;

  if (ARefNum <= Length(gArr) - 1) and Progress.isRunning then
  begin
    if (CmdLine.Params.Indexof(IntToStr(ARefNum)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      Progress.Log(RPad('Process List #' + IntToStr(ARefNum) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ARefNum].Count) + '):', RPAD_VALUE) + gArr[ARefNum].fi_Name_Program + SPACE + gArr[ARefNum].fi_Name_Program_Type + RUNNING);
      if gArr[ArefNum].fi_Process_ID <> '' then  Progress.Log(RPad('Process ID:', RPAD_VALUE) + gArr[ARefNum].fi_Process_ID);
      if gArr[ARefNum].fi_Process_As <> '' then Progress.Log(RPad('fi_Process_As:', RPAD_VALUE) + gArr[ARefNum].fi_Process_As);
      Result := True;
    end;
  {$IF DEFINED (ISFEXGUI)}
  end
  else
  begin
    if not Progress.isRunning then
      Exit;
    Progress.Log('Error: RefNum > ' + IntToStr(Length(gArr) - 1)); // noslz
  {$IFEND}
  end;
end;

// -----------------------------------------------------------------------------
// Total Validated File Count
// -----------------------------------------------------------------------------
function TotalValidatedFileCountInTLists: integer;
var
  i: integer;
begin
  Result := 0;
  for i := 0 to Length(gArr) - 1 do
  begin
    if not Progress.isRunning then
      break;
    Result := Result + gArr_ValidatedFiles_TList[i].Count;
  end;
end;

// =============================================================================
// Do Process
// =============================================================================
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);
const
  // DO NOT TRANSLATE - Property Names -----------------------------------------
  PROP_SSID_STR = 'SSID_STR'; // noslz
  PROP_LAST_JOINED = 'lastJoined'; // noslz
  PROP_LAST_AUTO_JOINED = 'lastAutoJoined'; // noslz
  PROP_BSSID = 'BSSID'; // noslz
var
  aArtifactEntry: TEntry;
  ADDList: TList;
  aPropertyTree: TPropertyParent;
  aRootProperty: TPropertyNode;
  BytesToStrings_StringList: TStringList;
  CarvedData: TByteInfo;
  CarvedEntry: TEntry;
  carved_str: string;
  ColCount: integer;
  col_DF: TDataStoreFieldArray;
  Display_Name_str: string;
  DNT_sql_col: string;
  end_pos: int64;
  FooterProgress: TPAC;
  HeaderReader, FooterReader, CarvedEntryReader: TEntryReader;
  HeaderRegex, FooterRegEx: TRegEx;
  h_startpos, h_offset, h_count, f_offset, f_count: int64;
  g, i, x, y, z: integer;
  Item: PSQL_FileSearch;
  mydb: TSQLite3Database;
  newEntryReader: TEntryReader;
  newJOURNALReader: TEntryReader;
  newWALReader: TEntryReader;
  Node1, Node2, Node3, Node4, Node5, Name_Node: TPropertyNode;
  NodeList1: TObjectList;
  NumberOfNodes: integer;
  PropertyList: TObjectList;
  PropertyList_WifiConfig: TObjectList;
  Re: TDIPerlRegEx;
  records_read_int: integer;
  sqlselect: TSQLite3Statement;
  sql_row_count: integer;
  temp_str: string;
  TotalFiles: int64;
  variant_Array: array of variant;

  // Special Conversions
  aduration_str: string;
  blob_bytes: TBytes;
  countrycode_int: integer;
  countrycode_str: string;
  Direction_str: string;
  duration_int: integer;
  networkcode_int: integer;
  networkcode_str: string;
  phonenumber_str: string;
  phonenumber_trunc_str: string;
  SSIDSTR: string;
  type_str: string;

  procedure NullTheArray;
  var
    idx: integer;
  begin
    for idx := 1 to ColCount do
      variant_Array[idx] := null;
  end;

// ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼
  procedure AddToModule;
  var
    NEntry: TArtifactItem;
    IsAllEmpty: boolean;
  begin

    // Do not add when it is a Triage
    if CmdLine.Params.Indexof(TRIAGE) > -1 then
      Exit;

     // Check if all columns are empty
    IsAllEmpty := True;

    for g := 1 to ColCount do
    begin
      if (not VarIsEmpty(variant_Array[g])) and (not VarIsNull(variant_Array[g])) then
      begin
        IsAllEmpty := False;
        break;
      end;
    end;

    // If artifacts are found
    if not IsAllEmpty then
    begin

      // Create the Category Folder to the tree --------------------------------
      if not assigned(gArtConnect_CatFldr) then
      begin
        gArtConnect_CatFldr := AddArtifactCategory(nil, CATEGORY_NAME, -1, gArr[1].fi_Icon_Category); { Sort index, icon }
        gArtConnect_CatFldr.Status := gArtConnect_CatFldr.Status + [dstUserCreated];
      end;

      // Create artifact sub-folder
      gArtConnect_ProgFldr[ref_num] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
      Item.fi_Name_Program,
      Item.fi_Name_Program_Type,
      Item.fi_Name_OS,
      Item.fi_Icon_Program,
      Item.fi_Icon_OS);

      // If the artifact sub-folder has been created
      if assigned(gArtConnect_ProgFldr[ref_num]) then
      begin
        // Set the status of the artifacts sub-folder
        gArtConnect_ProgFldr[ref_num].Status := gArtConnect_ProgFldr[ref_num].Status + [dstUserCreated];

        // Setup the columns of the artifact sub-folder
        SetUpColumnforFolder(ref_num, gArtConnect_ProgFldr[ref_num], col_DF, ColCount, aItems);

        // Create the new entry
        NEntry := TArtifactItem.Create;
        NEntry.SourceEntry := aArtifactEntry;
        NEntry.Parent := gArtConnect_ProgFldr[ref_num];
        NEntry.PhysicalSize := 0;
        NEntry.LogicalSize := 0;

        // Populate the columns
        try
          for g := 1 to ColCount do
          begin
            if not Progress.isRunning then
              break;

            if (VarIsNull(variant_Array[g])) or (VarIsEmpty(variant_Array[g])) then
              Continue;

            case col_DF[g].FieldType of
              ftDateTime:
                try
                  col_DF[g].AsDateTime[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftDateTime conversion');
                  end;
                end;

              ftFloat:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsFloat[NEntry] := StrToFloat(variant_Array[g]);
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftFloat conversion');
                    end;
                  end;

              ftInteger:
                try
                  if Trim(variant_Array[g]) = '' then
                    variant_Array[g] := null
                  else
                    col_DF[g].AsInteger[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftInteger conversion');
                  end;
                end;

              ftLargeInt:
                try
                  col_DF[g].AsInt64[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftLargeInt conversion');
                  end;
                end;

              ftString:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsString[NEntry] := variant_Array[g];
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftString conversion');
                    end;
                  end;

              ftBytes:
                try
                  col_DF[g].AsBytes[NEntry] := variantToArrayBytes(variant_Array[g]);
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftBytes conversion');
                  end;
                end;

            end;
          end;
        except
          on e: exception do
          begin
            Progress.Log(e.message);
          end;
        end;
        ADDList.Add(NEntry);
      end;
    end;
    NullTheArray;
  end;
// ▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲

  procedure Process_IOS_Voicemail_Greeting(anEntry: TEntry);
  begin
    for g := 1 to ColCount do
    begin
      if not Progress.isRunning then break;
      DNT_sql_col := aItems[g].sql_col;
      if (DNT_sql_col = 'DNT_FILENAME') and (aItems[g].read_as = ftString) then
      variant_Array[g] := anEntry.EntryName;
    end;
    AddToModule;
  end;

  function Test_SQL_Tables(refnum: integer): boolean;
  var
    LineList: TStringList;
    idx: integer;
    sql_tbl_count: integer;
    sqlselect_row: TSQLite3Statement;
    s: integer;
    temp_sql_str: string;

  begin
    Result := True;

    // Table count check to eliminate corrupt SQL files with no tables
    sql_tbl_count := 0;
    try
      sqlselect := TSQLite3Statement.Create(mydb, 'SELECT name FROM sqlite_master WHERE type=''table''');
      try
        while sqlselect.Step = SQLITE_ROW do
        begin
          if not Progress.isRunning then
            break;
          sql_tbl_count := sql_tbl_count + 1;
        end;
      finally
        FreeAndNil(sqlselect);
      end;
    except
      Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
      Result := False;
      Exit;
    end;

    if sql_tbl_count = 0 then
    begin
      Result := False;
      Exit;
    end;

    // Table count check to eliminate SQL files without required tables
    if (sql_tbl_count > 0) and (gArr[refnum].fi_SQLTables_Required <> '') then
    begin
      LineList := TStringList.Create;
      LineList.Delimiter := ','; // Default, but ";" is used with some locales
      LineList.StrictDelimiter := True; // Required: strings are separated *only* by Delimiter
      LineList.Sorted := True;
      LineList.Duplicates := dupIgnore;
      try
        LineList.DelimitedText := gArr[refnum].fi_SQLTables_Required;
        if LineList.Count > 0 then
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[0]) + '''))';
          for idx := 0 to LineList.Count - 1 do
            temp_sql_str := temp_sql_str + ' or (upper(name) = upper(''' + Trim(LineList[idx]) + '''))';

          temp_sql_str := temp_sql_str + ')';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) and (sqlselect_row.ColumnInt(0) = LineList.Count) then
                sql_tbl_count := 1
              else
              begin
                Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + aArtifactEntry.EntryName, RPAD_VALUE) + 'Ignored (Found ' + IntToStr(sqlselect_row.ColumnInt(0)) + ' of ' + IntToStr(LineList.Count) + ' required tables).');
                Result := False;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            MessageUser(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
            Result := False;
          end;
        end;

        // Log the missing required tables
        for s := 0 to LineList.Count - 1 do
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[s]) + ''')))';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) then
              begin
                if sqlselect_row.ColumnInt(0) = 0 then
                begin
                  Progress.Log(RPad('', RPAD_VALUE) + HYPHEN + 'Missing table:' + SPACE + Trim(LineList[s]));
                end;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
          end;
        end;

      finally
        LineList.free;
      end;
    end;

    // Row Count the matched table
    if sql_tbl_count > 0 then
    begin
      sql_row_count := 0;
      sqlselect_row := TSQLite3Statement.Create(mydb, 'SELECT COUNT(*) FROM ' + gArr[refnum].fi_SQLPrimary_Tablestr);
      try
        while sqlselect_row.Step = SQLITE_ROW do
        begin
          sql_row_count := sqlselect_row.ColumnInt(0);
          break;
        end;
      finally
        sqlselect_row.free;
      end;
    end;

  end;

  // ---------------------------------------------------------------------------
  // Protobuf Decode
  // ---------------------------------------------------------------------------
  function Decode_Protobuf(pEntry: TEntry): string;
  var
    DataStore_FS: TDataStore;
    DNT_sql_col: string;
    int64_timestamp1: int64;
    int64_timestamp2: int64;
    nEntry: TEntry;
    pbr: TGDProtoBufReader;
    pBytes: TBytes;
    str_file_id: string;
    str_real_activity: string;
    str_screenshot_pth: string;
    str_search_query: string;
    str_service: string;
    str_snapshot_fle: string;
    str_snapshot_pth: string;
    temp_dt: TDateTime;

  begin
    Result := '';
    if assigned(pEntry) and (pEntry.LogicalSize > 0) and assigned(NewEntryReader) then
    begin
      pBytes := NewEntryReader.AsBytes(pEntry.LogicalSize);

      // MOB_AND_GOOGLE_QUICKSEARCH - https://www.swiftforensics.com/2020/03/google-search-personal-assistant-data.html
      if (Item^.fi_Process_ID = 'MOB_AND_GOOGLE_QUICKSEARCH') then
      begin
        if UpperCase(pEntry.EntryName) = 'RECENTSDATASTORE.PB' then // noslz
        begin
          pbr := TGDProtoBufReader.Create;
          try
            pbr.Loadfrombytes(pBytes);
            for g := 1 to
              ColCount do variant_Array[g] := null; // Null the array
            str_file_id        := pbr.ProBufLookupValue(pbr.rootcell, [1], [1]);
            str_search_query   := pbr.ProBufLookupValue(pbr.rootcell, [1], [5]);
            str_service        := pbr.ProBufLookupValue(pbr.rootcell, [1.3], [5]);
            int64_timestamp1   := StrToInt64(trim(pbr.ProBufLookupValue(pbr.rootcell, [1], [4])));
            int64_timestamp2   := StrToInt64(trim(pbr.ProBufLookupValue(pbr.rootcell, [1.4], [17])));
            // Find first screenshot file by path ------------------------------
            DataStore_FS := GetDataStore(DATASTORE_FILESYSTEM);
            if DataStore_FS = nil then
              Exit;
            try
              str_screenshot_pth := '';
              nEntry := DataStore_FS.FindByPath(nEntry, '**\*' + str_file_id + '.jpg');
              if nEntry <> nil then
                str_screenshot_pth := nEntry.FullPathName;
            finally
              DataStore_FS.free;
            end;
            // Populate the columns --------------------------------------------
            for g := 1 to ColCount do
            begin
              if not Progress.isRunning then break;
              DNT_sql_col := aItems[g].sql_col;
              if (DNT_sql_col = 'DNT_TIMESTAMP') and (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftDateTime) then
              try
                temp_dt := UnixTimeToDateTime(int64_timestamp1 div 1000);
                variant_Array[g] := temp_dt;
              except
                on e: exception do
                begin
                  Progress.Log(e.message);
                end;
              end;
              if (DNT_sql_col = 'DNT_FILE_ID') then variant_Array[g] := str_file_id;
              if (DNT_sql_col = 'DNT_SEARCH_QUERY') then variant_Array[g] := str_search_query;
              if (DNT_sql_col = 'DNT_SCREENSHOT_PATH') then variant_Array[g] := str_screenshot_pth;
            end;
            AddToModule;
          finally
            pbr.free;
          end;
        end;
      end;

      // MOB_AND_LOGS_SYSTEMCE_SNAPSHOTS
      if (Item^.fi_Process_ID = 'MOB_AND_LOGS_SYSTEMCE_SNAPSHOTS') then // noslz
      begin
        if RegexMatch(pEntry.EntryName,'^\d{1,}.proto*', false) then // noslz
        begin
          pbr := TGDProtoBufReader.Create;
          try
            pbr.Loadfrombytes(pBytes);
            for g := 1 to
              ColCount do variant_Array[g] := null; // Null the array
            str_real_activity := pbr.ProBufLookupValue(pbr.rootcell, [], [10]);
            str_real_activity := StringReplace(str_real_activity, '"', '', [rfReplaceAll]);
            // Find first screenshot file by path ------------------------------
            DataStore_FS := GetDataStore(DATASTORE_FILESYSTEM);
            if DataStore_FS = nil then
              Exit;
            try
              str_snapshot_pth := '';
              str_snapshot_fle := '';
              str_file_id := StringReplace(pEntry.EntryName, pEntry.Extension, '.jpg', [rfReplaceAll]);
              nEntry := DataStore_FS.FindByPath(nEntry, '**\' + str_file_id);
              if nEntry <> nil then
              begin
                str_snapshot_pth := nEntry.FullPathName;
                str_snapshot_fle := nEntry.EntryName;
              end;
            finally
              DataStore_FS.free;
            end;
            // Populate the columns --------------------------------------------
            for g := 1 to ColCount do
            begin
              if not Progress.isRunning then break;
              DNT_sql_col := aItems[g].sql_col;
              if (DNT_sql_col = 'DNT_REAL_ACTIVITY') then variant_Array[g] := str_real_activity;
              if (DNT_sql_col = 'DNT_SNAPSHOT_FILE') then variant_Array[g] := str_snapshot_fle;
              if (DNT_sql_col = 'DNT_SNAPSHOT_FULLPATH') then variant_Array[g] := str_snapshot_pth;
            end;
            AddToModule;
          finally
            pbr.free;
          end;
        end;
      end;
    end;
  end;

  procedure AndroidBluetooth();
  var
    aStringList: TStringList;
    LineList: TStringList;
    tmp_str: string;
    x: integer;
  begin
    aStringList := TStringList.Create;
    LineList := TStringList.Create;
    try
      LineList.Delimiter := '='; // noslz
      LineList.StrictDelimiter := True;
      tmp_str := newEntryReader.AsPrintableChar(aArtifactEntry.LogicalSize);
      tmp_str := StringReplace(tmp_str, '.', CR, [rfReplaceAll]); // noslz
      aStringList.Text := tmp_str;
      for x := aStringList.Count - 1 downto 0 do
      begin
        if (Trim(aStringList[x]) = '') or (RegexMatch(aStringList[x], '\[', False)) then // noslz
          aStringList.Delete(x);
      end;
      for x := aStringList.Count - 1 downto 0 do
      begin
        LineList.DelimitedText := aStringList[x];
        if LineList.Count = 2 then
        begin
          for g := 1 to ColCount do
          begin
            DNT_sql_col := aItems[g].sql_col;
            delete(DNT_sql_col, 1, 4);
            if trim(UpperCase(LineList[0])) = UpperCase(DNT_sql_col) then
            begin
              variant_array[g] := trim(LineList[1]);
              Break;
            end;
          end;
        end;
      end;
    finally
      aStringList.free;
      LineList.free;
    end;
    AddToModule;
  end;

  procedure AndroidSettingsSecure();
  var
    file_text_str: string;
  begin
    file_text_str := '';
    try
      file_text_str := newEntryReader.AsPrintableChar(aArtifactEntry.LogicalSize);
    except
      Exit;
    end;
    if trim(file_text_str) <> '' then
    begin
      for g := 1 to ColCount do
      begin
        if Not Progress.isRunning then
          break;
        DNT_sql_col := aItems[g].sql_col;
        delete(DNT_sql_col, 1, 4);
        try
          variant_Array[g] := PerlMatch('(?<=' + DNT_sql_col + '\/)([^"]*)(?=\/)', file_text_str); // noslz
          variant_Array[g] := StringReplace(variant_Array[g], '.', '', [rfReplaceAll]);
        except
        end;
      end;
      AddToModule;
    end;
  end;

  procedure AndroidShutdownCheckpoints();
  var
    aStringList: TStringList;
    x: integer;
  begin
    aStringList := TStringList.Create;
    try
      if assigned(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
      begin
        try
          aStringList.LoadFromStream(newEntryReader);
        except
          Exit;
        end;
        for x := 0 to aStringList.Count - 1 do
        begin
          if RegexMatch(aStringList[x], 'Shutdown request from', True) then // noslz
          begin
            for g := 1 to ColCount do
            begin
              DNT_sql_col := aItems[g].sql_col;
              if DNT_sql_col = 'DNT_TIMESTAMP_STR' then variant_Array[g] := trim(PerlMatch('(?<= at)(.*)(?=\(epoch)', aStringList[x])); // noslz
              if DNT_sql_col = 'DNT_REQUESTOR'     then variant_array[g] := trim(PerlMatch('(?<=Shutdown request from)(.*)(?=for|at)', aStringList[x])); // noslz
              if DNT_sql_col = 'DNT_ENTRY'         then variant_array[g] := aStringList[x];
            end;
          end;
          AddToModule;
        end;
      end;
    finally
      aStringList.free;
    end;
  end;

  procedure AndroidVersion();
  var
    android_bld_str: string;
    android_cde_str: string;
    android_ver_str: string;
    LineList: TStringList;
    SubLineList: TStringList;
  begin
    LineList := TStringList.Create;
    SubLineList := TStringList.Create;
    SubLineList.Delimiter := '.';
    SubLineList.StrictDelimiter := True;
    try
      android_bld_str := '';
      android_cde_str := '';
      android_ver_str := '';
      LineList.Delimiter := ';';
      LineList.StrictDelimiter := True;
      LineList.DelimitedText := newEntryReader.AsPrintableChar(aArtifactEntry.LogicalSize);
      if LineList.Count = 3 then
      begin
        SubLineList.DelimitedText := LineList(0);
        if SubLineList.Count = 2 then
        begin
          android_ver_str := SubLineList(1);
        end;
        android_cde_str := LineList(1);
        android_bld_str := StringReplace(LineList(2), '.', '', [rfReplaceAll]);
        for g := 1 to ColCount do
        begin
          DNT_sql_col := aItems[g].sql_col;
          if DNT_sql_col = 'MOB_AND_LOGS_VERSION' then variant_array[g] := android_ver_str;
          if DNT_sql_col = 'DNT_BUILD_VERSION' then variant_array[g] := android_bld_str;
          if DNT_sql_col = 'DNT_CODENAME' then variant_array[g] := android_cde_str;
        end;
      end;
    finally
      LineList.free;
      SubLineList.free;
    end;
    AddToModule;
  end;

// Start of Do Process =========================================================
var
  idx: integer;
  message_length: integer;
  NSStr_pos: integer;
  process_proceed_bl: boolean;
  temp_flt: Double;
  temp_process_counter: integer;
  tmp_i64: int64;
  tmp_str: string;

begin
  if gArtifactsDataStore = nil then
    Exit;

  Item := @gArr[ref_num];
  temp_process_counter := 0;
  ColCount := LengthArrayTABLE(aItems);
  if (gArr_ValidatedFiles_TList[ref_num].Count > 0) then
  begin
    begin
      process_proceed_bl := True;
      if process_proceed_bl then
      begin
        SetLength(variant_Array, ColCount + 1);
        ADDList := TList.Create;
        newEntryReader := TEntryReader.Create;
        try
          Progress.Max := gArr_ValidatedFiles_TList[ref_num].Count;
          Progress.DisplayMessageNow := 'Process' + SPACE + PROGRAM_NAME + ' - ' + Item.fi_Name_OS + RUNNING;
          Progress.CurrentPosition := 1;

          // Regex Setup -------------------------------------------------------
          CarvedEntryReader := TEntryReader.Create;
          FooterProgress := TPAC.Create;
          FooterProgress.Start;
          FooterReader := TEntryReader.Create;
          FooterRegEx := TRegEx.Create;
          FooterRegEx.CaseSensitive := True;
          FooterRegEx.Progress := FooterProgress;
          FooterRegEx.SearchTerm := Item.fi_Carve_Footer;
          HeaderReader := TEntryReader.Create;
          HeaderRegex := TRegEx.Create;
          HeaderRegex.CaseSensitive := True;
          HeaderRegex.Progress := Progress;
          HeaderRegex.SearchTerm := Item.fi_Carve_Header;

          if HeaderRegex.LastError <> 0 then
          begin
            Progress.Log('HeaderRegex Error: ' + IntToStr(HeaderRegex.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          if FooterRegEx.LastError <> 0 then
          begin
            Progress.Log(RPad('!!!!!!! FOOTER REGEX ERROR !!!!!!!:', RPAD_VALUE) + IntToStr(FooterRegEx.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          TotalFiles := gArr_ValidatedFiles_TList[ref_num].Count;
          // Loop Validated Files ----------------------------------------------
          for i := 0 to gArr_ValidatedFiles_TList[ref_num].Count - 1 do { addList is freed a the end of this loop }
          begin
            if not Progress.isRunning then
              break;

            Progress.IncCurrentprogress;
            temp_process_counter := temp_process_counter + 1;
            Display_Name_str := GetFullName(Item);
            Progress.DisplayMessageNow := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + RUNNING;
            aArtifactEntry := TEntry(gArr_ValidatedFiles_TList[ref_num].items[i]);

            if assigned(aArtifactEntry) and newEntryReader.OpenData(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
            begin
              if BL_USE_FLAGS then aArtifactEntry.Flags := aArtifactEntry.Flags + [Flag8]; // Gray Flag = Process Routine

              // ===============================================================
              // Process As Custom
              // ===============================================================
              if UpperCase(Item^.fi_Process_As) = 'PROCESS_AS_CUSTOM' then
              begin
                if Item^.fi_Process_ID = 'MOB_IOS_VOICEMAIL_GREETING' then
                  Process_IOS_Voicemail_Greeting(aArtifactEntry);
              end;

              // ===============================================================
              // Process As Text
              // ===============================================================
              if UpperCase(Item^.fi_Process_As) = PROCESS_AS_TEXT then
              begin
                if Item^.fi_Process_ID = 'MOB_AND_BLUETOOTH_ADAPTER' then AndroidBluetooth;
                if Item^.fi_Process_ID = 'MOB_AND_LOGS_SETTINGS_SECURE' then AndroidSettingsSecure;
                if Item^.fi_Process_ID = 'MOB_AND_LOGS_SHUTDOWN_CHECKPOINTS' then AndroidShutdownCheckpoints;
                if Item^.fi_Process_ID = 'MOB_AND_LOGS_VERSION' then AndroidVersion;
              end;

              // ===============================================================
              // PROCESS AS: Nodes - .PropName, .PropDisplayValue, .PropValue, .PropDataType
              // ===============================================================
              if ((UpperCase(Item^.fi_Process_As) = PROCESS_AS_PLIST) or (UpperCase(Item^.fi_Process_As) = PROCESS_AS_XML)) and
              ((UpperCase(Item^.fi_Signature_Parent) = 'PLIST (BINARY)') or
              (UpperCase(Item^.fi_Signature_Parent) = 'JSON') or
              (UpperCase(Item^.fi_Signature_Parent) = 'XML') or
              (UpperCase(Item^.fi_Signature_Parent) = 'ABX')) then
              begin
                aPropertyTree := nil;
                try
                  if ProcessMetadataProperties(aArtifactEntry, aPropertyTree, newEntryReader) and assigned(aPropertyTree) then
                  begin
                    // Set Root Property (Level 0)
                    aRootProperty := aPropertyTree;
                    // Progress.Log(format('%-21s %-26s %-10s %-10s %-20s ',['Number of root child nodes:',IntToStr(aRootProperty.PropChildList.Count),'Bates:',IntToStr(aArtifactEntry.ID),aArtifactEntry.EntryName]));
                    if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                    begin
                      if not Progress.isRunning then
                        break;

                      // Locate NodeByName
                      if CompareText(aRootProperty.PropName, Item.fi_NodeByName) = 0 then
                        Node1 := aRootProperty
                      else
                        Node1 := aRootProperty.GetFirstNodeByName(Item.fi_NodeByName);

                      if assigned(Node1) then
                      begin

                        // Count the number of nodes
                        NodeList1 := Node1.PropChildList;
                        if assigned(NodeList1) then
                          NumberOfNodes := NodeList1.Count
                        else
                          NumberOfNodes := 0;
                        Progress.Log(RPad('Number of child nodes:', RPAD_VALUE) + IntToStr(NumberOfNodes));

                        // Apple CommandCenter iOS -----------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_APPLE_COMMANDCENTER') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          Progress.Log(aArtifactEntry.EntryName + SPACE + inttostr(aArtifactEntry.ID));
                          if assigned(aRootProperty) and (aRootProperty.PropName = 'PList (Binary)') then
                          begin
                            for g := 1 to ColCount do
                              variant_Array[g] := null;
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                              Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
                              if assigned(Node1) then
                                variant_Array[g] := Node1.PropDisplayValue;
                            end;
                            AddToModule;
                          end;
                        end;

                        // Application Data Folder - Process the nodes ---------
                        if (UpperCase(Item^.fi_Process_ID) = 'DNT_APPLICATION_FOLDER_UUID_IOS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          if assigned(aArtifactEntry.Parent) then
                          begin
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := aItems[g].sql_col;
                              if UpperCase(DNT_sql_col) = 'DNT_APPLICATION_NAME' then variant_Array[g] := Node1.PropDisplayValue;
                              if UpperCase(DNT_sql_col) = 'DNT_UUID' then variant_Array[g] := aArtifactEntry.Parent.EntryName;
                            end;
                            AddToModule;
                          end;
                        end;

                        // BlueTool Devices PList - Process the nodes ----------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_BLUETOOTH_DEVICES') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          if (UpperCase(Item^.fi_Signature_Parent) = 'PLIST (BINARY)') then
                          begin
                            for x := 0 to aRootProperty.PropChildList.Count - 1 do
                            begin
                              if not Progress.isRunning then
                                break;
                              Node1 := TPropertyNode(aRootProperty.PropChildList.items[x]);
                              if assigned(Node1) then
                              begin
                                PropertyList := Node1.PropChildList;
                                if assigned(PropertyList) then
                                  for y := 0 to PropertyList.Count - 1 do
                                  begin
                                    Node2 := TPropertyNode(PropertyList.items[y]);
                                    if assigned(Node2) then
                                    begin
                                      variant_Array[1] := Node1.PropName;
                                      for g := 1 to ColCount do
                                      begin
                                        DNT_sql_col := aItems[g].sql_col;
                                        delete(DNT_sql_col, 1, 4);
                                        if (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) then
                                          variant_Array[g] := Node2.PropDisplayValue;
                                      end;
                                    end;
                                  end;
                                AddToModule;
                              end;
                            end;
                          end;
                        end;

                        // iTunes Backup Details - XML - Process the nodes -----
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_ITUNES_XML') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          if (UpperCase(Item^.fi_Signature_Parent) = 'XML') then
                          begin
                            for x := 0 to NumberOfNodes - 1 do
                            begin
                              if not Progress.isRunning then
                                break;
                              Node1 := TPropertyNode(NodeList1.items[x]);
                              for g := 1 to ColCount do
                              begin
                                DNT_sql_col := aItems[g].sql_col;
                                delete(DNT_sql_col, 1, 4);
                                if assigned(Node1) and (UpperCase(Node1.PropDisplayValue) = UpperCase(DNT_sql_col)) then
                                begin
                                  Node1 := TPropertyNode(NodeList1.items[x + 1]);
                                  if assigned(Node1) then
                                  begin
                                    Progress.Log(RPad(' ' + Node1.PropName + ': ', RPAD_VALUE) + Node1.PropDisplayValue);
                                    variant_Array[g] := Node1.PropDisplayValue;
                                  end;
                                end;
                              end;
                            end;
                            AddToModule;
                          end;
                        end;

                        // iTunes PList - Process the nodes --------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_ITUNES_PLIST') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          if (UpperCase(Item^.fi_Signature_Parent) = 'PLIST (BINARY)') then
                          begin
                            for x := 0 to aRootProperty.PropChildList.Count - 1 do
                            begin
                              if not Progress.isRunning then
                                break;
                              Node1 := TPropertyNode(aRootProperty.PropChildList.items[x]);
                              if assigned(Node1) then
                              begin
                                for g := 1 to ColCount do
                                begin
                                  DNT_sql_col := aItems[g].sql_col;
                                  delete(DNT_sql_col, 1, 4);
                                  if (UpperCase(Node1.PropName) = UpperCase(DNT_sql_col)) then
                                  begin
                                    // Progress.Log(RPad(' ' + Node1.PropName + ': ', RPAD_VALUE) + Node1.PropDisplayValue);
                                    variant_Array[g] := Node1.PropDisplayValue;
                                  end;
                                end;
                              end;
                            end;
                            AddToModule;
                          end;
                        end;

                        // MAPS - Process the Nodes ----------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'DNT_MAPS_HISTORY') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          for x := 0 to NumberOfNodes - 1 do
                          begin
                            if not Progress.isRunning then
                              break;
                            Node1 := TPropertyNode(NodeList1.items[x]);
                            PropertyList := Node1.PropChildList;
                            if assigned(PropertyList) then
                              for y := 0 to PropertyList.Count - 1 do
                              begin
                                Node2 := TPropertyNode(PropertyList.items[y]);
                                if assigned(Node2) then
                                begin
                                  for g := 1 to ColCount do
                                  begin
                                    if Not Progress.isRunning then
                                      break;
                                    DNT_sql_col := aItems[g].sql_col;
                                    delete(DNT_sql_col, 1, 4);
                                    if assigned(Node2) and (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) then
                                    begin
                                      // Progress.Log(RPad(' ' + Node2.PropName + ': ', RPAD_VALUE) + Node2.PropDisplayValue);
                                      variant_Array[g] := Node2.PropDisplayValue;
                                    end;
                                  end;
                                end;
                              end;
                            AddToModule;
                          end;
                        end;

                        // Mobile Backup iOS - Process the Nodes ----------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_RESTORE_INFO') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          if not Progress.isRunning then
                            break;
                          Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item^.fi_NodeByName));
                          if assigned(Node1) then
                          begin
                            PropertyList := Node1.PropChildList;
                            if assigned(PropertyList) and (PropertyList.Count > 0) then
                              for y := 0 to PropertyList.Count - 1 do
                              begin
                                Node2 := TPropertyNode(PropertyList.items[y]);
                                if assigned(Node2) then
                                begin
                                  for g := 1 to ColCount do
                                  begin
                                    if Not Progress.isRunning then
                                      break;
                                    DNT_sql_col := aItems[g].sql_col;
                                    delete(DNT_sql_col, 1, 4);
                                    if assigned(Node2) and (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) then
                                    begin
                                      //Progress.Log(RPad(' ' + Node2.PropName + ': ', RPAD_VALUE) + Node2.PropDisplayValue);
                                      variant_Array[g] := Node2.PropDisplayValue;
                                    end;
                                  end;
                                end;
                              end;
                            AddToModule;
                          end;
                        end;

                        // Resent Tasks - Android ------------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_AND_LOGS_RECENT_TASKS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
                        begin
                          Progress.Log(aArtifactEntry.EntryName + SPACE + inttostr(aArtifactEntry.ID));
                          if assigned(aRootProperty) and (aRootProperty.PropName = 'ABX') then
                          begin
                            for g := 1 to ColCount do
                              variant_Array[g] := null;
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                              Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
                              if assigned(Node1) then
                              begin
                                if {(col_DF[g].FieldType = ftDateTime) and} (aItems[g].convert_as = 'UNIX_MS') then
                                try
                                  variant_Array[g] := Int64ToDateTime_ConvertAs(StrToInt64(Node1.PropDisplayValue),'UNIX_MS');
                                except
                                  on e: exception do
                                  begin
                                    Progress.Log(e.message);
                                  end;
                                end
                                else
                                  variant_Array[g] := Node1.PropDisplayValue;
                              end;
                            end;
                            AddToModule;
                          end;
                        end;

                        // Roles.xml - Android ---------------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_AND_LOGS_ROLES_XML') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
                        begin
                          Progress.Log(aArtifactEntry.EntryName + SPACE + inttostr(aArtifactEntry.ID));
                          if assigned(aRootProperty) and (aRootProperty.PropName = 'XML') then
                          begin
                            for g := 1 to ColCount do
                              variant_Array[g] := null; // Null the array

                            Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName('roles')); // noslz
                            if assigned(Node1) then
                            begin
                              PropertyList := Node1.PropChildList;

                              if assigned(PropertyList) then
                              begin
                                for y := 0 to PropertyList.Count - 1 do
                                begin
                                  Node2 := TPropertyNode(PropertyList.items[y]);

                                  if assigned(Node2) then
                                  begin
                                    Node3 := (TPropertyNode(Node2.GetFirstNodeByName('name'))); // noslz
                                    if assigned(Node3) then
                                    begin
                                      for g := 1 to ColCount do
                                      begin
                                        DNT_sql_col := aItems[g].sql_col;
                                        if (DNT_sql_col = 'DNT_ROLE') then
                                          variant_Array[g] := Node3.PropDisplayValue; // noslz
                                      end;
                                    end;

                                    Node4 := (TPropertyNode(Node2.GetFirstNodeByName('holder'))); // noslz
                                    if assigned(Node4) then
                                    begin
                                      Node5 := (TPropertyNode(Node4.GetFirstNodeByName('name'))); // noslz
                                      if assigned(Node5) then
                                      begin
                                        for g := 1 to ColCount do
                                        begin
                                          DNT_sql_col := aItems[g].sql_col;
                                          if (DNT_sql_col = 'DNT_HOLDER') then // noslz
                                            variant_Array[g] := Node5.PropDisplayValue;
                                        end;
                                      end;
                                    end;

                                  end;
                                  AddToModule;
                                end;
                              end;
                            end;
                          end;
                        end;

                        // Uber - JSON Accounts ---------------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_UBER_ACCOUNTS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          if assigned(aRootProperty) and (aRootProperty.PropName = 'JSON Tree') then
                          begin
                            for g := 1 to ColCount do
                              variant_Array[g] := null; // Null the array
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := aItems[g].sql_col;
                              delete(DNT_sql_col, 1, 4);
                              Node2 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
                              if assigned(Node2) then
                              begin
                                // String
                                if aItems[g].read_as = ftString then
                                  variant_Array[g] := Node2.PropDisplayValue;
                                // Date Time
                                if (aItems[g].col_type = ftDateTime) and (Node2.PropDataType = 'UString') then
                                  variant_Array[g] := Int64ToDateTime_ConvertAs(StrToInt64(Node2.PropValue), aItems[g].convert_as);
                              end;
                            end;
                            AddToModule;
                          end;
                        end;

                        // Uber - JSON EyeBall ---------------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'DNT_UBER EYEBALL') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          for g := 1 to ColCount do
                          begin
                            DNT_sql_col := aItems[g].sql_col;
                            delete(DNT_sql_col, 1, 4);
                            Node2 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
                            if assigned(Node2) then
                            begin

                              // String
                              if aItems[g].read_as = ftString then
                                variant_Array[g] := Node2.PropDisplayValue;
                              // Date Time
                              if (aItems[g].col_type = ftDateTime) and (Node2.PropDataType = 'UString') then
                                variant_Array[g] := Int64ToDateTime_ConvertAs(StrToInt64(Node2.PropValue), aItems[g].convert_as);
                            end;
                          end;
                          AddToModule;
                        end;

                        // WIFI ANDROID XML - Process the Nodes ----------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_AND_WIFI_CONFIGURATIONS_XML') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          for x := 0 to NumberOfNodes - 1 do
                          begin
                            Progress.Log(StringOfChar('-', CHAR_LENGTH));
                            Progress.Log('Processing' + SPACE + IntToStr(x + 1) + ':');
                            if not Progress.isRunning then
                              break;
                            Node1 := TPropertyNode(NodeList1.items[x]);
                            if assigned(Node1) and (Node1.PropName = 'Network') then // noslz
                            begin
                              PropertyList := Node1.PropChildList;
                              if assigned(PropertyList) then
                              begin
                                for y := 0 to PropertyList.Count - 1 do
                                begin
                                  Node2 := TPropertyNode(PropertyList.items[y]);
                                  if assigned(Node2) and (Node2.PropName = 'WifiConfiguration') then // noslz
                                  begin
                                    PropertyList_WifiConfig := Node2.PropChildList;
                                    if assigned(PropertyList_WifiConfig) then
                                    begin
                                      for z := 0 to PropertyList_WifiConfig.Count - 1 do
                                      begin
                                        Node3 := TPropertyNode(PropertyList_WifiConfig.items[z]);
                                        if Node3.PropName = 'string' then // noslz
                                        begin
                                          Name_Node := Node3.GetFirstNodeByName('name'); // noslz
                                          if assigned(Name_Node) then
                                          begin
                                            //Progress.Log(RPad(Name_Node.PropValue, RPAD_VALUE) + Node3.PropDisplayValue);
                                            for g := 1 to ColCount do
                                            begin
                                              DNT_sql_col := aItems[g].sql_col;
                                              delete(DNT_sql_col, 1, 4);
                                              if DNT_sql_col = UpperCase(Name_Node.PropValue) then
                                              begin
                                                temp_str := StringReplace(Node3.PropDisplayValue, '"', ' ', [rfReplaceAll]);
                                                variant_Array[g] := Trim(temp_str);
                                                //Progress.Log(RPad(Name_Node.PropValue, RPAD_VALUE) + Node3.PropDisplayValue);
                                              end;
                                            end;
                                          end;
                                        end;
                                      end;
                                      AddToModule;
                                    end;
                                  end;
                                end;
                              end;
                            end;

                          end; { number of nodes }
                        end; { wifi }

                        // WIFI HOTSPOT ANDROID XML - Process the Nodes --------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_AND_WIFI_CONFIGURATIONS_XML_HOTSPOT') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          for x := 0 to NumberOfNodes - 1 do
                          begin
                            if not Progress.isRunning then break;
                            Node1 := TPropertyNode(NodeList1.items[x]);
                            if assigned(Node1) then // noslz
                            begin
                              PropertyList := Node1.PropChildList;
                              if assigned(PropertyList) then
                              begin
                                for y := 0 to PropertyList.Count - 1 do
                                begin
                                  Node2 := TPropertyNode(PropertyList.items[y]);
                                  if assigned(Node2) and (Node2.PropName = 'name') and RegexMatch(Node2.PropDisplayValue,'SSID|Passphrase', True) then // noslz
                                  begin
                                    for g := 1 to ColCount do
                                    begin
                                      DNT_sql_col := aItems[g].sql_col;
                                      if DNT_sql_col = 'DNT_SSID' then
                                        variant_array[g] := Node1.PropDisplayValue;
                                      if DNT_sql_col = 'DNT_PASSPHRASE' then
                                        variant_array[g] := Node1.PropDisplayValue;
                                    end;
                                  end;
                                end;
                              end;
                            end;
                          end;
                          AddToModule;
                        end;

                        // WIFI iOS - Process the Nodes ------------------------
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_WIFI_CONFIGURATIONS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array
                          for x := 0 to NumberOfNodes - 1 do
                          begin
                            if not Progress.isRunning then
                              break;
                            SSIDSTR := '';
                            Node1 := TPropertyNode(NodeList1.items[x]);
                            PropertyList := Node1.PropChildList;
                            if assigned(PropertyList) then
                              for y := 0 to PropertyList.Count - 1 do
                              begin
                                Node2 := TPropertyNode(PropertyList.items[y]);
                                if assigned(Node2) then
                                begin
                                  for g := 1 to ColCount do
                                  begin
                                    if not Progress.isRunning then
                                      break;
                                    DNT_sql_col := aItems[g].sql_col;
                                    delete(DNT_sql_col, 1, 4);
                                    if assigned(Node2) and (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) then
                                    begin
                                      if (Node2.PropName = PROP_LAST_JOINED) and (aItems[g].convert_as = 'VARTODT') and (aItems[g].col_type = ftDateTime) then
                                        try
                                          variant_Array[g] := vartodatetime(Node2.PropValue);
                                        except
                                          Progress.Log(ATRY_EXCEPT_STR + 'An exception occurred for PROP_LAST_JOINED:      ' + Node2.PropDisplayValue);
                                        end
                                      else if (Node2.PropName = PROP_LAST_AUTO_JOINED) and (aItems[g].convert_as = 'VARTODT') and (aItems[g].col_type = ftDateTime) then
                                        try
                                          variant_Array[g] := vartodatetime(Node2.PropValue);
                                        except
                                          Progress.Log(ATRY_EXCEPT_STR + 'An exception occurred for PROP_LAST_AUTO_JOINED: ' + Node2.PropDisplayValue);
                                        end
                                      else
                                        variant_Array[g] := Node2.PropDisplayValue;
                                    end;
                                  end;
                                end;
                              end;
                            AddToModule;
                          end; { number of nodes }
                        end; { wifi }

                        // Wifi Networks iOS
                        if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_WIFI_KNOWN_NETWORKS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
                        begin
                          for g := 1 to ColCount do
                            variant_Array[g] := null; // Null the array

                          for x := 0 to aRootProperty.PropChildList.Count - 1 do
                          begin
                            Node1 := TPropertyNode(aRootProperty.PropChildList.items[x]);
                            if assigned(Node1) then
                            begin
                              for g := 1 to ColCount do
                              begin
                                if not Progress.isRunning then break;
                                DNT_sql_col := aItems[g].sql_col;
                                delete(DNT_sql_col, 1, 4);

                                if RegexMatch(Node1.PropName, DNT_sql_col, false) then
                                  variant_Array[g] := StringReplace(Node1.PropName, 'wifi.network.ssid.', '', [rfReplaceAll]); // noslz

                                if DNT_sql_col = 'BSSID' then // noslz
                                begin
                                  Node2 := TPropertyNode(Node1.GetFirstNodeByName('BSSID')); // noslz
                                  if assigned(Node2) then
                                    variant_Array[g] := Node2.PropDisplayValue;
                                end;

                                if DNT_sql_col = 'JOINEDBYUSERAT' then // noslz
                                begin
                                  Node2 := TPropertyNode(Node1.GetFirstNodeByName('JoinedByUserAt')); // noslz
                                  if assigned(Node2) and (Node2.PropName = 'JoinedByUserAt') and (aItems[g].convert_as = 'VARTODT') and (aItems[g].col_type = ftDateTime) then // noslz
                                  try
                                    variant_Array[g] := vartodatetime(Node2.PropValue);
                                  except
                                    Progress.Log(ATRY_EXCEPT_STR + SPACE + 'JOINEDBYUSERAT'); // noslz
                                  end
                                end;

                              end;

                            end;
                            AddToModule;
                          end;
                        end;

                      end; { assigned Node1 }
                    end
                    else
                      Progress.Log((format('%-39s %-10s %-10s', ['Could not open data:', '-', 'Bates: ' + IntToStr(aArtifactEntry.ID) + ' ' + aArtifactEntry.EntryName])));
                  end;
                finally
                  if assigned(aPropertyTree) then
                    FreeAndNil(aPropertyTree);
                end;
              end;

              // ===============================================================
              // PROCESS AS: Protobuf
              // ===============================================================
              if (Item^.fi_Process_ID = 'MOB_AND_GOOGLE_QUICKSEARCH') then // noslz
                Decode_Protobuf(aArtifactEntry);

              if (Item^.fi_Process_ID = 'MOB_AND_LOGS_SYSTEMCE_SNAPSHOTS') then // noslz
                Decode_Protobuf(aArtifactEntry);

              // ===============================================================
              // PROCESS AS: SQL
              // ===============================================================
              if RegexMatch(Item.fi_Signature_Parent, RS_SIG_SQLITE, False) then
              begin
                newWALReader := GetWALReader(gFileSystemDataStore, aArtifactEntry);
                newJOURNALReader := GetJOURNALReader(gFileSystemDataStore, aArtifactEntry);
                try
                  mydb := TSQLite3Database.Create;
                  try
                    mydb.OpenStream(newEntryReader, newJOURNALReader, newWALReader);

                    if Test_SQL_Tables(ref_num) then
                    begin
                      records_read_int := 0;

                      // ***************************************************************************************************************************
                      sqlselect := TSQLite3Statement.Create(mydb, (Item.fi_SQLStatement));
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      //Progress.Log(Item.fi_SQLStatement);
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      // ***************************************************************************************************************************

                      Progress.Log(RPad('SQL Row Count:', RPAD_VALUE) + inttostr(sql_row_count));

                      while sqlselect.Step = SQLITE_ROW do
                      begin
                        if not Progress.isRunning then
                          break;
                        records_read_int := records_read_int + 1;

                        // Progress for large files
                        if sql_row_count > 15000 then
                        begin
                          Progress.DisplayMessages := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + SPACE +
                            IntToStr(records_read_int) + '/' + IntToStr(sql_row_count) + RUNNING;
                        end;

                        // Read the values from the SQL tables -----------------
                        for g := 1 to ColCount do
                        begin
                          if not Progress.isRunning then
                            break;
                          DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                          if aItems[g].read_as = ftString then
                            variant_Array[g] := ColumnValueByNameAsText(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftinteger) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftLargeInt) then
                            variant_Array[g] := ColumnValueByNameAsint64(sqlselect, DNT_sql_col)

                          else if (aItems[g].col_type = ftDateTime) then
                          begin
                            tmp_i64 := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                            if tmp_i64 > 0 then
                            begin
                              try
                                if aItems[g].read_as = ftFloat then
                                begin
                                  temp_flt := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                                  variant_Array[g] := temp_flt;
                                  variant_Array[g] := GHFloatToDateTime(variant_Array[g], aItems[g].convert_as);
                                end
                                else
                                begin
                                  variant_Array[g] := Int64ToDateTime_ConvertAs(tmp_i64, aItems[g].convert_as);
                                end;
                              except
                                on e: exception do
                                  Progress.Log(e.message);
                              end;
                            end;
                          end;

                          // Add the table name and row location
                          if DNT_sql_col = 'SQLLOCATION' then
                            variant_Array[g] := 'Table: ' + lowercase(Item.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz

                          // SPECIAL CONVERSIONS FOLLOW ========================

                          //  SMS v4
                          if Item^.fi_Process_ID = 'SMS_V4_IOS' then // noslz
                          begin
                            if DNT_sql_col = 'attributedBody' then // noslz
                            begin
                              blob_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              try
                                if (Length(blob_bytes)) > 12 then
                                begin
                                  if BytesToString(blob_bytes, 2, 12) = 'streamtyped' then // noslz - Blob header
                                  begin
                                    NSStr_pos := BytePos_Of_HexPattern('4E53537472696E67', blob_bytes); // noslz - "NSString"
                                    NSStr_pos := NSStr_pos + 12;
                                    if BytesToHex_Start_Length(blob_bytes, NSStr_pos, 2) = '2B 81' then // noslz - Cater for multiple bytes, including the length byte, after the +
                                    begin
                                      NSStr_pos := NSStr_pos + 2;
                                      message_length := blob_bytes[NSStr_pos];
                                      NSStr_pos := NSStr_pos + 1;
                                      if RegexMatch(BytesToHex_Start_Length(blob_bytes, NSStr_pos, 1),'00|01', false) then
                                      begin
                                        NSStr_pos := NSStr_pos + 1;
                                        variant_Array[g] := UTF8BytesToString(blob_bytes, NSStr_pos, message_length);
                                      end;
                                    end
                                    else
                                    begin // Single length byte after the +
                                      NSStr_pos := NSStr_pos + 1;
                                      message_length := blob_bytes[NSStr_pos];
                                      NSStr_pos := NSStr_pos + 1;
                                      variant_Array[g] := UTF8BytesToString(blob_bytes, NSStr_pos, message_length);
                                    end;
                                  end;
                                end;
                              finally
                                blob_bytes := nil;
                              end;
                            end;
                          end;

                          // APPLE_MAPS
                          if Item^.fi_Process_ID = 'DNT_APPLE_MAPS' then // noslz
                          begin
                            //  ZMAPITEMSTORAGE
                            if (UpperCase(DNT_sql_col) = 'ZMAPITEMSTORAGE') then // noslz
                            begin
                              blob_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              try
                                if (Length(blob_bytes)) > 1 then
                                  variant_Array[g] := AppleMaps_ProtoBuf(blob_bytes, 'ZMAPITEMSTORAGE'); // noslz - Process the ProtoBuf blob
                              finally
                                blob_bytes := nil;
                              end;
                            end;
                            // ZROUTEREQUESTSTORAGE
                            if (UpperCase(DNT_sql_col) = 'ZROUTEREQUESTSTORAGE') then // noslz
                            begin
                              blob_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              try
                                if (Length(blob_bytes)) > 10 then
                                  variant_Array[g] := AppleMaps_ProtoBuf(blob_bytes,'ZROUTEREQUESTSTORAGE'); // noslz - Process the ProtoBuf blob
                              finally
                                blob_bytes := nil;
                              end;
                            end;
                          end;

                           // Photos Android 12 --------------------------------
                          // https://thebinaryhick.blog/2020/10/19/androids-external-db-everything-old-is-new-again/
                          if (Item^.fi_Name_Program = 'Photos' + SPACE + ANDROID) and (Item^.fi_Name_Program_Type = '12') then // noslz
                          begin
                            if (UpperCase(DNT_sql_col) = 'FORMAT') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0 = data (file or folder)'
                              else if (variant_Array[g] = '12288') then
                                variant_Array[g] := '12288 = file'
                              else if (variant_Array[g] = '12289') then
                                variant_Array[g] := '12289 = folder'
                              else if (variant_Array[g] = '14336') then
                                variant_Array[g] := '14336 = picture'
                              else if (variant_Array[g] = '47360') then
                                variant_Array[g] := '47360 = audio'
                              else if (variant_Array[g] = '47488') then
                                variant_Array[g] := '14336 = video'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'IS_DOWNLOAD') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0 = No'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1 = Yes'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'MEDIA_TYPE') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0 = data (file or folder)'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1 = picture'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := '2 = auido'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := '3 = video'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                          end;

                          // Photos iOS 14 -------------------------------------
                          // https://github.com/ScottKjr3347/iOS_Local_PL_Photos.sqlite_Queries/blob/main/iOS14/iOS14_LPL_Phsql_Locations.txt
                          if Item^.fi_Name_Program = PHOTOS_SQLITE then // noslz
                          begin
                            if (UpperCase(DNT_sql_col) = 'ZCLOUDDELETESTATE') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-Not deleted'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-Deleted'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZFAVORITE') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-No'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-Yes'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZHIDDEN') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-Asset Not Hidden'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-Asset Hidden'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZKIND') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-Photo'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-Video'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZKINDSUBTYPE') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-Still-Photo'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := '2-Live-Photo'
                              else if (variant_Array[g] = '10') then
                                variant_Array[g] := '10-SpringBoard-Screenshot'
                              else if (variant_Array[g] = '100') then
                                variant_Array[g] := '100-Video'
                              else if (variant_Array[g] = '101') then
                                variant_Array[g] := '101-Slow-Mo-Video'
                              else if (variant_Array[g] = '102') then
                                variant_Array[g] := '102-Time-lapse-Video'
                              else if (variant_Array[g] = '103') then
                                variant_Array[g] := '103-Replay_Screen_Recording'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZLATITUDE') then // noslz
                              if (variant_Array[g] = '-180.0') then
                                variant_Array[g] := '';

                            if (UpperCase(DNT_sql_col) = 'ZLONGITUDE') then // noslz
                              if (variant_Array[g] = '-180.0') then
                                variant_Array[g] := '';

                            if (UpperCase(DNT_sql_col) = 'ZORIENTATION') then // noslz
                            begin
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-Video-Default_Adjustment_Horizontal-Camera-(left)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := '2-Horizontal-Camera-(right)'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := '3-Horizontal-Camera-(right)'
                              else if (variant_Array[g] = '4') then
                                variant_Array[g] := '4-Horizontal-Camera-(left)'
                              else if (variant_Array[g] = '5') then
                                variant_Array[g] := '5-Vertical-Camera-(top)'
                              else if (variant_Array[g] = '6') then
                                variant_Array[g] := '6-Vertical-Camera-(top)'
                              else if (variant_Array[g] = '7') then
                                variant_Array[g] := '7-Vertical-Camera-(bottom)'
                              else if (variant_Array[g] = '8') then
                                variant_Array[g] := '8-Vertical-Camera-(bottom)'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                            if (UpperCase(DNT_sql_col) = 'ZTRASHEDSTATE') then // noslz
                            begin
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := '0-Not In Trash or Recently Deleted'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := '1-In Trash or Recently Deleted'
                              else
                                variant_Array[g] := 'unknown';
                            end;

                          end;

                          // Android Call Logs ---------------------------------
                          if UpperCase(Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type) = 'CALL LOGS' then
                          begin
                            // Direction
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'TYPE') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Incoming'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Outgoing'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := 'Missed'
                              else
                                variant_Array[g] := 'Unknown';
                          end;

                          // Calls ---------------------------------------------
                          if UpperCase(Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type) = 'CALLS' then
                          begin
                            // Clean up Number
                            if ((DNT_sql_col = 'Address') or (DNT_sql_col = 'ZADDRESS')) and (aItems[g].fex_col = 'Number') and (aItems[g].col_type = ftString) then
                            begin
                              phonenumber_str := '';
                              phonenumber_trunc_str := '';
                              phonenumber_str := trim(variant_Array[g]);
                              phonenumber_trunc_str := phonenumber_str;
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, '#', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, '+', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, '-', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, '(', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, '}', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, ' ', '', [rfReplaceAll]);
                              phonenumber_trunc_str := StringReplace(phonenumber_trunc_str, ' ', '', [rfReplaceAll]);
                              variant_Array[g] := phonenumber_trunc_str;
                            end;

                            // Duration
                            if ((DNT_sql_col = 'ZDURATION')) and (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            begin
                              duration_int := variant_Array[g];
                              aduration_str := '';
                              aduration_str := FormatDateTime('hh:mm:ss', duration_int / secsperday);
                              variant_Array[g] := aduration_str;
                            end;

                            // Direction
                            Direction_str := '';
                            if (DNT_sql_col = 'ZORIGINATED') then
                              if (variant_Array[g] = '0') then
                              begin
                                variant_Array[g] := 'Incoming';
                                Direction_str := 'Incoming';
                              end
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Outgoing'
                              else
                                variant_Array[g] := 'Unknown';
                            // Type of Call
                            if (DNT_sql_col = 'Flags') then
                              if (variant_Array[g] = '0') or (variant_Array[g] = '4') then
                                variant_Array[g] := 'Incoming'
                              else if (variant_Array[g] = '8') then
                                variant_Array[g] := 'Blocked'
                              else if (variant_Array[g] = '16') then
                                variant_Array[g] := 'Incoming - Facetime'
                              else if (variant_Array[g] = '1') or (variant_Array[g] = '5') or (variant_Array[g] = '9') or (variant_Array[g] = '17') or (variant_Array[g] = '21') then
                                variant_Array[g] := 'Outgoing'
                              else
                                variant_Array[g] := '-';
                            // Country Code
                            if (DNT_sql_col = 'country_code') // noslz
                              and (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            begin
                              countrycode_str := '';
                              countrycode_int := variant_Array[g];
                              countrycode_str := MMC_Loookup_Country(countrycode_int);
                              variant_Array[g] := countrycode_str;
                            end;
                            // Network Code
                            if (DNT_sql_col = 'network_code') // noslz
                              and (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            begin
                              networkcode_str := '';
                              networkcode_int := variant_Array[g];
                              networkcode_str := MMC_Loookup_Network(countrycode_int, networkcode_int);
                              variant_Array[g] := networkcode_str;
                            end;
                          end; { Calls }

                          // IOS Call History ----------------------------------
                          if UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_CALLS_12' then // noslz
                          begin

                            // Answered
                            if (DNT_sql_col = 'ZANSWERED') then // noslz
                            begin
                              tmp_str := trim(variant_Array[g]);
                              if tmp_str = '0' then
                                variant_Array[g] := 'No';
                              if tmp_str = '1' then
                                variant_Array[g] := 'Yes';
                            end;

                            // Call Type
                            if (DNT_sql_col = 'ZCALLTYPE') then // noslz
                            begin
                              tmp_str := trim(variant_Array[g]);
                              if tmp_str = '0' then
                                variant_Array[g] := 'Third-Party App';
                              if tmp_str = '1' then
                                variant_Array[g] := 'Phone Call';
                              if tmp_str = '8' then
                                variant_Array[g] := 'FaceTime Video';
                              if tmp_str = '16' then
                                variant_Array[g] := 'FaceTime Audio';
                            end;

                            // Direction
                            if (DNT_sql_col = 'ZORIGINATED') then // noslz
                            begin
                              tmp_str := trim(variant_Array[g]);
                              if tmp_str = '0' then
                                variant_Array[g] := 'Incoming';
                              if tmp_str = '1' then
                                variant_Array[g] := 'Outgoing';
                            end;

                            // Disconnection Cause
                            if (DNT_sql_col = 'ZDISCONNECTED_CAUSE') then // noslz
                            begin
                              tmp_str := trim(variant_Array[g]);
                              if tmp_str = '0' then
                                variant_Array[g] := 'Ended';
                              if tmp_str = '6' then
                                variant_Array[g] := 'Rejected';
                            end;
                          end;

                        end;
                        AddToModule;
                      end;
                      if assigned(sqlselect) then
                        sqlselect.free;
                      Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + copy(aArtifactEntry.EntryName, 1, 40), RPAD_VALUE) + format('%-5s %-5s', [IntToStr(records_read_int), '(SQL)']));
                    end;
                  finally
                    FreeAndNil(mydb);
                    if assigned(newWALReader) then
                      FreeAndNil(newWALReader);
                    if assigned(newJOURNALReader) then
                      FreeAndNil(newJOURNALReader);
                  end;
                except
                  on e: exception do
                  begin
                    Progress.Log(ATRY_EXCEPT_STR + RPad('WARNING:', RPAD_VALUE) + 'ERROR DOING SQL QUERY!');
                    Progress.Log(e.message);
                  end;
                end;
              end;

              // ===============================================================
              // PROCESS AS: BYTES TO StringList
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_BYTES_STRLST) then
              begin
                // Process Dynamic Dictionary ==================================
                if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_DYNAMIC_DICTIONARY') then
                begin
                  if HeaderReader.opendata(aArtifactEntry) then
                    try
                      Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                      HeaderReader.Position := 24;
                      BytesToStrings_StringList := BytesToStringList(HeaderReader, HeaderReader.Size, 10000);
                      for idx := 0 to BytesToStrings_StringList.Count - 1 do
                      begin
                        variant_Array[1] := BytesToStrings_StringList[idx];
                        AddToModule;
                      end;
                      BytesToStrings_StringList.free;
                    except
                      Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + aArtifactEntry.EntryName);
                    end;
                end;
              end;

              // ===============================================================
              // PROCESS AS: REGEX CARVE
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_CARVE) then
              begin
                if HeaderReader.OpenData(aArtifactEntry) and FooterReader.OpenData(aArtifactEntry) then
                  try
                    Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                    h_startpos := 0;
                    HeaderReader.Position := 0;
                    HeaderRegex.Stream := HeaderReader;
                    FooterRegEx.Stream := FooterReader;
                    // Find the first match, h_offset returned is relative to start pos
                    HeaderRegex.Find(h_offset, h_count);
                    while h_offset <> -1 do // h_offset returned as -1 means no hit
                    begin
                      // Now look for a footer
                      FooterRegEx.Stream.Position := h_startpos + h_offset + Item.fi_Carve_Adjustment;
                      // Limit looking for the footer to our max size
                      end_pos := h_startpos + h_offset + MAX_CARVE_SIZE;
                      if end_pos >= HeaderRegex.Stream.Size then
                        end_pos := HeaderRegex.Stream.Size - 1; // Don't go past the end!
                      FooterRegEx.Stream.Size := end_pos;
                      FooterRegEx.Find(f_offset, f_count);
                      // Found a footer and the size was at least our minimum
                      if (f_offset <> -1) and (f_offset + f_count >= MIN_CARVE_SIZE) then
                      begin
                        // Footer found - Create an Entry for the data found
                        CarvedEntry := TEntry.Create;
                        // ByteInfo is data described as a list of byte runs, usually just one run
                        CarvedData := TByteInfo.Create;
                        // Point to the data of the file
                        CarvedData.ParentInfo := aArtifactEntry.DataInfo;
                        // Adds the block of data
                        CarvedData.RunLstAddPair(h_offset + h_startpos, f_offset + f_count);
                        CarvedEntry.DataInfo := CarvedData;
                        CarvedEntry.LogicalSize := CarvedData.Datasize;
                        CarvedEntry.PhysicalSize := CarvedData.Datasize;
                        begin
                          if CarvedEntryReader.OpenData(CarvedEntry) then
                          begin
                            CarvedEntryReader.Position := 0;
                            carved_str := CarvedEntryReader.AsPrintableChar(CarvedEntryReader.Size);
                            for g := 1 to ColCount do
                              variant_Array[g] := null; // Null the array
                            // Setup the PerRegex Searches
                            Re := TDIPerlRegEx.Create(nil);
                            Re.CompileOptions := [coCaseLess];
                            Re.SetSubjectStr(carved_str);

                            // Process Android Wifi Flattened Data =============
                            if (UpperCase(Item^.fi_Process_ID) = 'DNT_WIFI_FLATTEMED_DATA') then
                            begin
                              // SSID ------------------------------------------
                              Re.MatchPattern := '(?<=ssid=")(.*?)(?=")';
                              if Re.Match(0) >= 0 then
                              begin
                                variant_Array[2] := Re.matchedstr;
                                AddToModule;
                              end;
                            end;

                            // Process Google Maps =============================
                            if (UpperCase(Item^.fi_Process_ID) = 'MOB_IOS_GOOGLE_MAPS') then
                            begin
                              // NAME ------------------------------------------
                              Re.MatchPattern := '(?<=\"\.)(\w.*?)(?=\*\$)';
                              if Re.Match(0) >= 0 then
                                variant_Array[1] := Re.matchedstr;

                              // URL -------------------------------------------
                              Re.MatchPattern := '(?<=5\.{6}[a-zA-Z0-9])(.*?)(?=\.{2}\xFF\xFF\xFF)';
                              if Re.Match(0) >= 0 then
                                variant_Array[2] := Re.matchedstr;

                              // ADDRESS ---------------------------------------
                              Re.MatchPattern := '(?<="R.|Z\.)(.*?)(?=J\.\.\.\.)';
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := Re.matchedstr;
                                temp_str := StringReplace(temp_str, 'z(..', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, 'z,..', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, 'z5..', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, ':.', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '..', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '2.', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, 'Z.', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, 'B.', ', ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, 'R.', ', ', [rfReplaceAll]);
                                variant_Array[3] := temp_str;
                              end;
                              if (variant_Array[1] <> '') or (variant_Array[2] <> '') or (variant_Array[3] <> '') then
                                AddToModule;
                            end;
                            if assigned(Re) then
                              FreeAndNil(Re);
                          end;
                        end;
                      end;
                      HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match
                    end;
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + aArtifactEntry.EntryName);
                  end; { Regex Carve }
              end;
            end;
            Progress.IncCurrentprogress;
          end;

          // Add to the gArtifactsDataStore
          if assigned(ADDList) and Progress.isRunning then
          begin
            if (CmdLine.Params.Indexof(TRIAGE) = -1) then
              gArtifactsDataStore.Add(ADDList);
            Progress.Log(RPad('Total Artifacts:', RPAD_VALUE) + IntToStr(ADDList.Count));

            // Export L01 files where artifacts are found (controlled by Artifact_Utils.pas)
            if (gArr_ValidatedFiles_TList[ref_num].Count > 0) and (ADDList.Count > 0) then
              ExportToL01(gArr_ValidatedFiles_TList[ref_num], Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type);

            ADDList.Clear;
          end;

          Progress.Log(StringOfChar('-', CHAR_LENGTH));
        finally
          records_read_int := 0;
          FreeAndNil(CarvedEntryReader);
          FreeAndNil(FooterProgress);
          FreeAndNil(FooterReader);
          FreeAndNil(FooterRegEx);
          FreeAndNil(HeaderReader);
          FreeAndNil(HeaderRegex);
          FreeAndNil(ADDList);
          FreeAndNil(newEntryReader);
        end;
      end;

    end;

  end
  else
  begin
    Progress.Log(RPad('', RPAD_VALUE) + 'No files to process.');
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
  end;

end;

// ==========================================================================================================================================================
// Start of Script
// ==========================================================================================================================================================
const
  CREATE_GLOB_SEARCH_BL = False;
  SEARCHING_FOR_ARTIFACT_FILES_STR = 'Searching for Artifact files';

var
  AboutToProcess_StringList: TStringList;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  FolderEntry: TEntry;
  AllFoundList_count: integer;
  AllFoundListUnique: TUniqueListOfEntries;
  DeleteFolder_display_str: string;
  DeleteFolder_TList: TList;
  Enum: TEntryEnumerator;
  ExistingFolders_TList: TList;
  FieldItunesDomain: TDataStoreField;
  FieldItunesName: TDataStoreField;
  FindEntries_StringList: TStringList;
  FoundList, AllFoundList: TList;
  GlobSearch_StringList: TStringList;
  i: integer;
  Item: TSQL_FileSearch;
  iTunes_Domain_str: string;
  iTunes_Name_str: string;
  Process_ID_str: string;
  ResultInt: integer;

{$IF DEFINED (ISFEXGUI)}
  Display_StringList: TStringList;
  MyScriptForm: TScriptForm;
{$IFEND}

procedure LogExistingFolders(aExistingFolders_TList: TList; aDeleteFolder_TList: TList); // Compare About to Process Folders with Existing Artifact Module Folders
const
  LOG_BL = False;
var
  aFolderEntry: TEntry;
  idx, t: integer;
begin
  if LOG_BL then Progress.Log('Existing Folders:');
  for idx := 0 to aExistingFolders_TList.Count - 1 do
  begin
    if not Progress.isRunning then break;
    aFolderEntry := (TEntry(aExistingFolders_TList[idx]));
    if LOG_BL then
      Progress.Log(HYPHEN + aFolderEntry.FullPathName);
    for t := 0 to AboutToProcess_StringList.Count - 1 do
    begin
      if aFolderEntry.FullPathName = AboutToProcess_StringList[t] then
        aDeleteFolder_TList.Add(aFolderEntry);
    end;
  end;
  if LOG_BL then
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

// Start of script -------------------------------------------------------------
var
  anEntry: TEntry;
  anint: integer;
  param_str: string;
  progress_program_str: string;
  ref_num: integer;
  regex_search_str: string;
  n, r, s: integer;
  temp_int: integer;
  test_param_int: integer;
  temp_process_counter: integer;

begin
  Read_SQLite_DB;

  SetLength(gArr_ValidatedFiles_TList, gNumberOfSearchItems);
  SetLength(gArtConnect_ProgFldr, gNumberOfSearchItems);

  Progress.Log(SCRIPT_NAME + ' started' + RUNNING);
  if (CmdLine.Params.Indexof(TRIAGE) > -1) then
    Progress.DisplayTitle := 'Triage' + HYPHEN + 'File System' + HYPHEN + CATEGORY_NAME
  else
    Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME;
  Progress.LogType := ltVerbose; // ltOff, ltVerbose, ltDebug, ltTechnical

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := ('Processing complete.');
    Exit;
  end;

  // Parameters Received -------------------------------------------------------
  param_str := '';
  test_param_int := 0;
  gParameter_Num_StringList := TStringList.Create;
  try
//    if CmdLine.ParamCount = 0 then
//    begin
//      MessageUser('No processing parameters received.');
//      Exit;
//    end
//    else
    begin
      Progress.Log(RPad('Parameters Received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
      for n := 0 to CmdLine.ParamCount - 1 do
      begin
        if not Progress.isRunning then
          break;

        param_str := param_str + '"' + CmdLine.Params[n] + '"' + ' ';
        // Validate Parameters
        if (RegexMatch(CmdLine.Params[n], '\d{1,2}$', False)) then
        begin
          try
            test_param_int := StrToInt(CmdLine.Params[n]);
            if (test_param_int <= -1) or (test_param_int > Length(gArr) - 1) then
            begin
              MessageUser('Invalid parameter received: ' + (CmdLine.Params[n]) + CR + ('Maximum is: ' + IntToStr(Length(gArr) - 1) + DCR + SCRIPT_NAME + ' will terminate.'));
              Exit;
            end;
            gParameter_Num_StringList.Add(CmdLine.Params[n]);
            Item := gArr[test_param_int];
            Progress.Log(RPad(HYPHEN + 'param_str ' + IntToStr(n) + ' = ' + CmdLine.Params[n], RPAD_VALUE) + format('%-10s %-10s %-25s %-12s', ['Ref#: ' + CmdLine.Params[n], CATEGORY_NAME,
              Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, Item.fi_Name_OS]));
          except
            begin
              Progress.Log(ATRY_EXCEPT_STR + 'Error validating parameter. ' + SCRIPT_NAME + ' will terminate.');
              Exit;
            end;
          end;
        end;
      end;
      Trim(param_str);
    end;
    Progress.Log(RPad('param_str:', RPAD_VALUE) + param_str); // noslz
    Progress.Log(StringOfChar('-', CHAR_LENGTH));

    // Progress Bar Text
    progress_program_str := '';
    if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      progress_program_str := PROGRAM_NAME;
    end;

    // Create GlobSearch -------------------------------------------------------
    if CREATE_GLOB_SEARCH_BL then
    begin
      GlobSearch_StringList := TStringList.Create;
      GlobSearch_StringList.Sorted := True;
      GlobSearch_StringList.Duplicates := dupIgnore;
      try
        for n := 0 to Length(gArr) - 1 do
          Create_Global_Search(n, gArr[n], GlobSearch_StringList);
        Progress.Log(GlobSearch_StringList.Text);
      finally
        GlobSearch_StringList.free;
      end;
      Exit;
    end;

    for n := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      if (CmdLine.Params.Indexof(IntToStr(n)) > -1) then
      begin
        Item := gArr[n];
        progress_program_str := 'Ref#:' + SPACE + param_str + SPACE + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type;
        break;
      end;
    end;

    if progress_program_str = '' then
      progress_program_str := 'Other';

    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;

    try
      gFileSystemDataStore := GetDataStore(DATASTORE_FILESYSTEM);
      if not assigned(gFileSystemDataStore) then
      begin
        Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + DCR + TSWT);
        Exit;
      end;

      try
        FieldItunesDomain := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_DOMAIN);
        FieldItunesName := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_NAME);

        // Create TLists For Valid Files
        for n := 0 to Length(gArr) - 1 do
        begin
          if not Progress.isRunning then
            break;
          gArr_ValidatedFiles_TList[n] := TList.Create;
        end;

        try
          AboutToProcess_StringList := TStringList.Create;
          try
            if (CmdLine.Params.Indexof('MASTER') = -1) then
            begin
              if (CmdLine.Params.Indexof('NOSHOW') = -1) then
              begin
                // PROCESSALL - Create the AboutToProcess_StringList
                if (CmdLine.Params.Indexof(PROCESSALL) > -1) or (CmdLine.ParamCount = 0) then
                begin
                  for n := 0 to Length(gArr) - 1 do
                  begin
                    if not Progress.isRunning then break;
                    Item := gArr[n];
                    AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[n]));
                    Progress.Log(format('%-4s %-59s %-30s', ['#' + IntToStr(n), Item.fi_Process_ID, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type]));
                  end;
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end
                else

                // PROCESS INDIVIDUAL - Create the AboutToProcess_StringList
                begin
                  if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
                  begin
                    for n := 0 to gParameter_Num_StringList.Count - 1 do
                    begin
                      if not Progress.isRunning then break;
                      temp_int := StrToInt(gParameter_Num_StringList[n]);
                      Item := gArr[temp_int];
                      AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[temp_int]));
                    end;
                  end;
                end;

                if CmdLine.ParamCount = 0 then
                begin
                  Progress.Log('No parameters received. Use a number or "PROCESSALL".');
                  Exit;
                end;

                // Show the form
                ResultInt := 1; // Continue AboutToProcess

{$IF DEFINED (ISFEXGUI)}
                Display_StringList := TStringList.Create;
                Display_StringList.Sorted := True;
                Display_StringList.Duplicates := dupIgnore;
                try
                  for i := 0 to AboutToProcess_StringList.Count - 1 do
                  begin
                    Display_StringList.Add(ExtractFileName(AboutToProcess_StringList[i]));
                  end;

                  if assigned(AboutToProcess_StringList) and (AboutToProcess_StringList.Count > 30) then
                  begin
                    MyScriptForm := TScriptForm.Create;
                    try
                      MyScriptForm.SetCaption(SCRIPT_DESCRIPTION);
                      MyScriptForm.SetText(Display_StringList.Text);
                      ResultInt := idCancel;
                      if MyScriptForm.ShowModal then
                        ResultInt := idOk
                    finally
                      FreeAndNil(MyScriptForm);
                    end;
                  end
                  else
                  begin
                    ResultInt := MessageBox('Extract Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME, (MB_OKCANCEL or MB_ICONQUESTION or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                  end;
                finally
                  Display_StringList.free;
                end;
{$IFEND}
              end;

              if ResultInt > 1 then
              begin
                Progress.Log(CANCELED_BY_USER);
                Progress.DisplayMessageNow := CANCELED_BY_USER;
                Exit;
              end;

            end;

            // Deal with Existing Artifact Folders
            ExistingFolders_TList := TList.Create;
            DeleteFolder_TList := TList.Create;
            try
              if gArtifactsDataStore.Count > 1 then
              begin
                anEntry := gArtifactsDataStore.First;
                while assigned(anEntry) and Progress.isRunning do
                begin
                  if anEntry.isDirectory then
                    ExistingFolders_TList.Add(anEntry);
                  anEntry := gArtifactsDataStore.Next;
                end;
                gArtifactsDataStore.Close;
              end;

              LogExistingFolders(ExistingFolders_TList, DeleteFolder_TList);

              // Create the delete folder TList and display string
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                Progress.Log('Replacing folders:');
                for s := 0 to DeleteFolder_TList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  FolderEntry := TEntry(DeleteFolder_TList[s]);
                  DeleteFolder_display_str := DeleteFolder_display_str + #13#10 + FolderEntry.FullPathName;
                  Progress.Log(HYPHEN + FolderEntry.FullPathName);
                end;
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;

              // Message Box - When artifact folder is already present ---------------------
{$IF DEFINED (ISFEXGUI)}
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                if (CmdLine.Params.Indexof('MASTER') = -1) and (CmdLine.Params.Indexof('NOSHOW') = -1) then
                begin

                  if (CmdLine.Params.Indexof('NOSHOW') = -1) then
                  begin
                    if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
                    begin
                      ResultInt := MessageBox('Artifacts have already been processed:' + #13#10 + DeleteFolder_display_str + DCR + 'Replace the existing Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME,
                        (MB_OKCANCEL or MB_ICONWARNING or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                    end;
                  end
                  else
                    ResultInt := idOk;
                  case ResultInt of
                    idOk:
                      begin
                        try
                          gArtifactsDataStore.Remove(DeleteFolder_TList);
                        except
                          MessageBox('ERROR: There was an error deleting existing artifacts.' + #13#10 + 'Save then reopen your case.', SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
                        end;
                      end;
                    idCancel:
                      begin
                        Progress.Log(CANCELED_BY_USER);
                        Progress.DisplayMessageNow := CANCELED_BY_USER;
                        Exit;
                      end;
                  end;
                end;
              end;
{$IFEND}

            finally
              ExistingFolders_TList.free;
              DeleteFolder_TList.free;
            end;
          finally
            AboutToProcess_StringList.free;
          end;

          // Find iTunes Backups and run Signature Analysis
          if (CmdLine.Params.Indexof('NOITUNES') = -1) then
            Itunes_Backup_Signature_Analysis;

          // Create the RegEx Search String
          regex_search_str := '';
          begin
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if (CmdLine.Params.Indexof(IntToStr(n)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                if Item.fi_Regex_Search <> ''      then regex_search_str := regex_search_str + '|' + Item.fi_Regex_Search;
                if Item.fi_Rgx_Itun_Bkup_Dmn <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Dmn;
                if Item.fi_Rgx_Itun_Bkup_Nme <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Nme;
              end;
            end;
          end;
          if (regex_search_str <> '') and (regex_search_str[1] = '|') then
            Delete(regex_search_str, 1, 1);

          AllFoundList := TList.Create;
          try
            AllFoundListUnique := TUniqueListOfEntries.Create;
            FoundList := TList.Create;
            FindEntries_StringList := TStringList.Create;
            FindEntries_StringList.Sorted := True;
            FindEntries_StringList.Duplicates := dupIgnore;
            try
              if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                Progress.Log('Find files by path (PROCESSALL)' + RUNNING);
                for i := low(gArr) to high(gArr) do
                begin
                  if (gArr[i].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob1_Search);
                  if (gArr[i].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob2_Search);
                  if (gArr[i].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob3_Search);
                end;
              end
              else
              begin
                Progress.Log('Find files by path (Individual)' + RUNNING);
                for i := 0 to gParameter_Num_StringList.Count - 1 do
                begin
                  anint := StrToInt(gParameter_Num_StringList[i]);
                  if (gArr[anint].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob1_Search);
                  if (gArr[anint].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob2_Search);
                  if (gArr[anint].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob3_Search);
                end;
              end;

              // Find the files by path and add to AllFoundListUnique
              Progress.Initialize(FindEntries_StringList.Count, STR_FILES_BY_PATH + RUNNING);
              gtick_foundlist_i64 := GetTickCount;
              for i := 0 to FindEntries_StringList.Count - 1 do
              begin
                if not Progress.isRunning then
                  break;
                try
                  Find_Entries_By_Path(gFileSystemDataStore, FindEntries_StringList[i], FoundList, AllFoundListUnique);
                except
                  Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Find_Entries_By_Path');
                end;
                Progress.IncCurrentprogress;
              end;

              Progress.Log(RPad(STR_FILES_BY_PATH + SPACE + '(Unique)' + COLON, RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Add > 40 char files with no extension to Unique List (possible iTunes backup files)
              Add40CharFiles(AllFoundListUnique);
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Move the AllFoundListUnique list into a TList
              if assigned(AllFoundListUnique) and (AllFoundListUnique.Count > 0) then
              begin
                Enum := AllFoundListUnique.GetEnumerator;
                while Enum.MoveNext do
                begin
                  anEntry := Enum.Current;
                  AllFoundList.Add(anEntry);
                end;
              end;

              // Now work with the TList from now on
              if assigned(AllFoundList) and (AllFoundList.Count > 0) then
              begin
                Progress.Log(RPad('Unique Files by path:', RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
                Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Add flags
                if BL_USE_FLAGS then
                begin
                  Progress.Log('Adding flags' + RUNNING);
                  for i := 0 to AllFoundList.Count - 1 do
                  begin
                    if not Progress.isRunning then
                      break;
                    anEntry := TEntry(AllFoundList[i]);
                    anEntry.Flags := anEntry.Flags + [Flag7];
                  end;
                  Progress.Log('Finished adding flags' + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Determine signature
                if assigned(AllFoundList) and (AllFoundList.Count > 0) then
                  SignatureAnalysis(AllFoundList);

                gtick_foundlist_str := (RPad(STR_FILES_BY_PATH + COLON, RPAD_VALUE) + CalcTimeTaken(gtick_foundlist_i64));
              end;

            finally
              FoundList.free;
              AllFoundListUnique.free;
              FindEntries_StringList.free;
            end;

            if assigned(AllFoundList) and (AllFoundList.Count > 0) then
            begin
              Progress.Log(SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              Progress.Log(format(GFORMAT_STR, ['', 'Action', 'Ref#', 'Bates', 'Signature', 'Filename (trunc)', 'Reason'])); // noslz
              Progress.Log(format(GFORMAT_STR, ['', '------', '----', '-----', '---------', '----------------', '------'])); // noslz

              AllFoundList_count := AllFoundList.Count;
              Progress.Initialize(AllFoundList_count, SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              for i := 0 to AllFoundList.Count - 1 do
              begin
                Progress.DisplayMessages := SEARCHING_FOR_ARTIFACT_FILES_STR + SPACE + '(' + IntToStr(i) + ' of ' + IntToStr(AllFoundList_count) + ')' + RUNNING;
                if not Progress.isRunning then
                  break;
                anEntry := TEntry(AllFoundList[i]);

                if (i mod 10000 = 0) and (i > 0) then
                begin
                  Progress.Log('Processing: ' + IntToStr(i) + ' of ' + IntToStr(AllFoundList.Count) + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Set the iTunes Domain String
                iTunes_Domain_str := '';
                if assigned(FieldItunesDomain) then
                begin
                  try
                    iTunes_Domain_str := FieldItunesDomain.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Domain string');
                  end;
                end;

                // Set the iTunes Name String
                iTunes_Name_str := '';
                if assigned(FieldItunesName) then
                begin
                  try
                    iTunes_Name_str := FieldItunesName.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Name string');
                  end;
                end;

                // Run the match
                aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
                if
                (anEntry.Extension <> 'db-shm') and
                (anEntry.Extension <> 'db-wal') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite WAL') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite SHM') then
                begin
                  if RegexMatch(anEntry.EntryName, regex_search_str, False) or
                  RegexMatch(anEntry.FullPathName, regex_search_str, False) or
                  FileSubSignatureMatch(anEntry) or
                  RegexMatch(iTunes_Domain_str, regex_search_str, False) or
                  RegexMatch(iTunes_Name_str, regex_search_str, False) then
                  begin

                    // Sub Signature Match
                    if FileSubSignatureMatch(anEntry) then
                    begin
                      if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag2]; // Blue Flag
                      DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                    end
                    else

                    // Regex Name Match
                    begin
                      if ((not anEntry.isDirectory) or (anEntry.isDevice)) and (anEntry.LogicalSize > 0) and (anEntry.PhysicalSize > 0) then
                      begin
                        if FileNameRegexSearch(anEntry, iTunes_Domain_str, iTunes_Name_str, regex_search_str) then
                        begin
                          if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag1]; // Red Flag
                          DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                        end;
                      end;
                    end;

                  end;
                  Progress.IncCurrentprogress;
                end;
              end;
            end;
          finally
            AllFoundList.free;
          end;

          // Check to see if files were found
          if (TotalValidatedFileCountInTLists = 0) then
          begin
            Progress.Log('No ' + PROGRAM_NAME + SPACE + 'files were found.');
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Not found';
          end
          else
          begin
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
            Progress.Log(RPad('Total Validated Files:', RPAD_VALUE) + IntToStr(TotalValidatedFileCountInTLists));
            Progress.Log(StringOfChar('=', CHAR_LENGTH + 80));
          end;

          // Display the content of the TLists for further processing
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.Log('Lists available to process' + RUNNING);
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if gArr_ValidatedFiles_TList[n].Count > 0 then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log(RPad('TList ' + IntToStr(n) + ': ' + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type + HYPHEN + Item.fi_Name_OS, RPAD_VALUE) + IntToStr(gArr_ValidatedFiles_TList[n].Count));
                for r := 0 to gArr_ValidatedFiles_TList[n].Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  anEntry := (TEntry(TList(gArr_ValidatedFiles_TList[n]).items[r]));
                  if not(Item.fi_Process_As = 'POSTPROCESS') then
                    Progress.Log(RPad(' Bates: ' + IntToStr(anEntry.ID), RPAD_VALUE) + anEntry.EntryName + SPACE + Item.fi_Process_As);
                end;
              end;
            end;
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
          end;

          // *** CREATE GUI ***
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Found';

            Progress.Log(RPad('Process All Lists' + RUNNING, RPAD_VALUE) + IntToStr(High(gArr) + 1)); // noslz
            Progress.Log(StringOfChar('=', CHAR_LENGTH));

            // *** DO PROCESS ***
            temp_process_counter := 0;
            Progress.CurrentPosition := 0;
            Progress.Max := TotalValidatedFileCountInTLists;
            gtick_doprocess_i64 := GetTickCount;

            // *****************************************************************
            for n := 0 to Length(gArr) - 1 do
            begin
              ref_num := n;
              if (TestForDoProcess(ref_num)) then
              begin
                Item := gArr[n];
                Process_ID_str := Item.fi_Process_ID;
                DoProcess(gArtConnect_ProgFldr[ref_num], ref_num, Mobile_Columns.GetTable(Process_ID_str));
              end;
            end;
            // *****************************************************************

            gtick_doprocess_str := (RPad('DoProcess:', RPAD_VALUE) + CalcTimeTaken(gtick_doprocess_i64));

          end;

          if not Progress.isRunning then
          begin
            Progress.Log(CANCELED_BY_USER);
            Progress.DisplayMessageNow := CANCELED_BY_USER;
          end;

        finally
          for n := 0 to Length(gArr) - 1 do
            FreeAndNil(gArr_ValidatedFiles_TList[n]);
        end;

      finally
        if assigned(gFileSystemDataStore) then
          FreeAndNil(gFileSystemDataStore);
      end;

    finally
      if assigned(gArtifactsDataStore) then
        FreeAndNil(gArtifactsDataStore);
    end;

  finally
    if assigned(gParameter_Num_StringList) then
      FreeAndNil(gParameter_Num_StringList);
  end;

  if Progress.isRunning then
  begin
    Progress.Log(gtick_foundlist_str);
    Progress.Log(gtick_doprocess_str);
  end;

  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  Progress.Log(SCRIPT_NAME + ' finished.');
  Progress.DisplayMessageNow := ('Processing complete.');

end.